# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## 项目概述

这是一个 Chrome 浏览器扩展,用于在即梦平台 (https://jimeng.jianying.com) 批量提交图片生成任务。本项目将原 Python 自动化脚本改造为浏览器插件,提供更好的用户体验。

## 核心架构

### 模块化设计

项目采用模块化架构,核心功能分离为独立的 JavaScript 模块:

```
content/modules/
├── storyboard-parser.js   # 分镜解析 (CSV/TSV/Story 三种格式)
├── role-extractor.js       # 角色提取 (正则匹配)
├── image-uploader.js       # 参考图上传 (DataTransfer API)
├── submit-handler.js       # 提交流程控制 (重试机制)
├── image-downloader.js     # 图片下载 (MutationObserver)
└── dom-helper.js          # DOM 工具函数
```

### 通信流程

1. **Popup → Content Script**: 通过 `chrome.tabs.sendMessage()` 发送任务参数
2. **Content Script → Background**: 通过 `chrome.runtime.sendMessage()` 请求下载图片
3. **Background → Downloads API**: 处理文件下载,避免 CORS 问题

### 关键技术点

- **Manifest V3**: 使用 Service Worker 替代 Background Page
- **MutationObserver**: 监听 DOM 变化检测提交状态和生成的图片
- **DataTransfer API**: 模拟拖拽上传参考图
- **PapaParse**: CSV 解析库,已本地化避免 CDN 依赖

## 常用开发命令

### 加载和调试扩展

1. **首次加载**:
   ```
   打开 chrome://extensions/
   启用"开发者模式"
   点击"加载已解压的扩展程序"
   选择项目根目录
   ```

2. **重新加载扩展** (代码修改后):
   ```
   chrome://extensions/ → 点击刷新图标
   或在扩展详情页点击"重新加载"
   ```

3. **调试各个部分**:
   - **Content Script**: 在即梦页面按 F12,查看 Console
   - **Popup**: 右键扩展图标 → "检查弹出内容"
   - **Background**: chrome://extensions/ → 点击 "Service Worker"

### 关键选择器位置

需要根据即梦网站实际 DOM 结构调整以下选择器:

- `submit-handler.js:138-143` - 提示词输入框选择器
- `submit-handler.js:197-202` - 提交按钮选择器
- `image-uploader.js:92-96` - 参考图上传区选择器
- `submit-handler.js:95-99` - 参考图项目选择器

## 工作流程理解

### 完整提交流程

对于每个分镜,`submit-handler.js` 的 `processOne()` 方法执行:

1. **清除旧参考图** - `clearReferenceImages()`
2. **提取角色** - 使用 `role-extractor.js`
3. **上传参考图** - 使用 `image-uploader.js`
4. **填入提示词** - `inputContent()`
5. **点击提交** - `clickSubmit()`
6. **检查状态** - `checkSubmitStatus()` (监听 Toast 提示)
7. **等待间隔** - 根据用户设置的延迟时间

### 重试机制

- 默认重试 2 次 (可在 UI 中配置)
- 任何步骤失败都会重新执行整个流程
- 超过最大重试次数后标记为失败

### 格式检测逻辑

`storyboard-parser.js` 智能检测三种格式:

- **CSV**: 检测引号、逗号、表头
- **TSV**: 检测制表符
- **Story**: 匹配 "故事1:"、"故事2:" 等分隔符

## 状态管理

### Chrome Storage

使用 `chrome.storage.local` 持久化:
- `batchLogs`: 最近 50 条日志 (滚动删除)
- 用户设置 (范围、间隔、重试次数)

### 全局状态变量

在 `content.js` 中:
- `isRunning`: 任务是否正在执行
- `isPaused`: 任务是否暂停
- `fileContent`: 当前加载的分镜文件内容
- `referenceFiles`: 用户选择的参考图文件列表

## 防止重复注入

使用全局标记防止 Content Script 重复加载:
```javascript
if (window.__JIMENG_CONTENT_INSTALLED) {
  throw new Error('插件已注入,禁止重复加载');
}
window.__JIMENG_CONTENT_INSTALLED = true;
```

## 悬浮窗版本说明

当前版本使用悬浮球 + 面板设计:
- 悬浮球可拖拽,默认位置右下角
- 点击悬浮球展开/收起面板
- 面板包含完整的文件上传、设置、日志功能
- 使用 `content.css` 定义样式,避免与页面样式冲突

## 图标文件要求

需要在 `icons/` 目录下放置:
- `icon16.png` (16x16)
- `icon48.png` (48x48)
- `icon128.png` (128x128)

## 常见问题处理

### 选择器失效

即梦网站更新可能导致选择器失效,需要:
1. 在即梦页面 F12 检查实际 DOM 结构
2. 更新对应模块中的选择器数组
3. 选择器数组按优先级排列,会依次尝试

### 提交状态检测

`checkSubmitStatus()` 监听 Toast 提示关键词:
- **成功**: "生成中"、"排队"、"已提交"、"成功"
- **失败**: "失败"、"错误"、"违规"、"频繁"

如果即梦修改提示文案,需要更新这些关键词列表。

### 参考图上传失败

常见原因:
1. 文件名与角色名不匹配
2. 上传区选择器失效
3. 即梦限制了拖拽上传

解决方法:
- 检查 `image-uploader.js` 的选择器
- 查看 Console 日志中的详细错误
- 考虑添加更多后备上传方式

## 开发注意事项

### 异步处理

所有 DOM 操作都要考虑异步加载:
- 使用 `await delay()` 给页面响应时间
- MutationObserver 监听动态内容

### 日志规范

使用统一的日志前缀方便调试:
```javascript
console.log('[模块名] 消息内容');
```

### 用户体验

- 所有操作提供视觉反馈 (进度条、日志)
- 失败时显示清晰的错误信息
- 关键操作前后添加适当延迟

### 浏览器兼容性

当前仅支持 Chrome,因为使用了 Chrome 专有 API:
- `chrome.downloads`
- `chrome.storage.local`
- `chrome.runtime.sendMessage`

要支持 Firefox 需要使用 `browser.*` API 的 polyfill。
