/**
 * Background Service Worker (Manifest V3)
 * 处理文件操作、下载管理和持久化存储
 */

console.log('[Background] Service Worker 已启动');

// ==================== 外部网站消息监听 ====================
chrome.runtime.onMessageExternal.addListener((message, sender, sendResponse) => {
  console.log('[Background] 收到外部消息:', message, '来自:', sender.url);

  // 导入 CSV 数据
  if (message.action === 'IMPORT_CSV') {
    console.log('[Background] 接收到 CSV 导入请求');
    const characterImages = message.characterImages || [];
    console.log(`[Background] 角色图片数量: ${characterImages.length}`);

    // 找到即梦标签页并发送数据
    chrome.tabs.query({ url: 'https://jimeng.jianying.com/*' }, (tabs) => {
      if (tabs.length === 0) {
        // 没有打开即梦页面，打开一个新的
        chrome.tabs.create({ url: 'https://jimeng.jianying.com/ai-tool/image/generate' }, (newTab) => {
          // 等待页面加载后发送数据
          chrome.storage.local.set({
            pendingCsvImport: message.csvData,
            pendingFileName: message.fileName || '外部导入.csv',
            pendingCharacterImages: characterImages
          }, () => {
            sendResponse({ success: true, message: '已打开即梦页面，数据将自动导入' });
          });
        });
      } else {
        // 已有即梦页面，直接发送数据
        chrome.tabs.sendMessage(tabs[0].id, {
          action: 'IMPORT_CSV_FROM_EXTERNAL',
          csvData: message.csvData,
          fileName: message.fileName || '外部导入.csv',
          characterImages: characterImages
        }, (response) => {
          if (chrome.runtime.lastError) {
            // 如果 content script 还没加载，暂存数据
            chrome.storage.local.set({
              pendingCsvImport: message.csvData,
              pendingFileName: message.fileName || '外部导入.csv',
              pendingCharacterImages: characterImages
            }, () => {
              sendResponse({ success: true, message: '数据已暂存，请刷新即梦页面' });
            });
          } else {
            sendResponse({ success: true, message: '数据已发送到即梦插件' });
          }
        });
        // 激活即梦标签页
        chrome.tabs.update(tabs[0].id, { active: true });
      }
    });

    return true; // 异步响应
  }

  // 返回插件 ID
  if (message.action === 'PING') {
    sendResponse({
      success: true,
      extensionId: chrome.runtime.id,
      version: chrome.runtime.getManifest().version
    });
    return true;
  }

  sendResponse({ success: false, message: '未知操作' });
  return true;
});

// ==================== 内部消息监听 ====================
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[Background] 收到消息:', message);

  // 处理下载请求
  if (message.action === 'download') {
    chrome.downloads.download({
      url: message.url,
      filename: message.filename,
      saveAs: false
    }, (downloadId) => {
      if (chrome.runtime.lastError) {
        console.error('[Background] 下载失败:', chrome.runtime.lastError);
        sendResponse({ success: false, error: chrome.runtime.lastError.message });
      } else {
        console.log('[Background] 下载ID:', downloadId);
        sendResponse({ success: true, downloadId: downloadId });
      }
    });
    return true;
  }

  // 处理图片获取 (避免 CORS)
  if (message.action === 'fetchImage') {
    fetch(message.url)
      .then(response => response.blob())
      .then(blob => {
        const reader = new FileReader();
        reader.onloadend = () => {
          sendResponse({ success: true, data: reader.result });
        };
        reader.readAsDataURL(blob);
      })
      .catch(error => {
        sendResponse({ success: false, error: error.message });
      });
    return true;
  }

  // 保存日志
  if (message.action === 'saveLog') {
    chrome.storage.local.get(['batchLogs'], (result) => {
      const logs = result.batchLogs || [];
      logs.push(message.log);
      if (logs.length > 1000) {
        logs.shift();
      }
      chrome.storage.local.set({ batchLogs: logs }, () => {
        sendResponse({ success: true });
      });
    });
    return true;
  }

  // 获取日志
  if (message.action === 'getLogs') {
    chrome.storage.local.get(['batchLogs'], (result) => {
      sendResponse({ logs: result.batchLogs || [] });
    });
    return true;
  }

  // 清空日志
  if (message.action === 'clearLogs') {
    chrome.storage.local.set({ batchLogs: [] }, () => {
      sendResponse({ success: true });
    });
    return true;
  }
});

// ==================== 扩展安装/更新 ====================
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    console.log('[Background] 扩展已安装');
    chrome.storage.local.set({
      batchLogs: [],
      settings: {
        startIndex: 1,
        endIndex: 999,
        delay: 2,
        retryTimes: 1,
        autoDownload: false
      }
    });
  } else if (details.reason === 'update') {
    console.log('[Background] 扩展已更新到版本:', chrome.runtime.getManifest().version);
  }
});

// 图标点击功能已移除（避免 API 兼容性问题）

// ==================== 下载完成监听 ====================
chrome.downloads.onChanged.addListener((delta) => {
  if (delta.state && delta.state.current === 'complete') {
    console.log('[Background] 下载完成:', delta.id);
  }
  if (delta.error) {
    console.error('[Background] 下载错误:', delta.error.current);
  }
});

// ==================== 定期清理 ====================
chrome.alarms.create('cleanupStorage', { periodInMinutes: 60 });

chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'cleanupStorage') {
    console.log('[Background] 执行存储清理');
    chrome.storage.local.get(['batchLogs'], (result) => {
      const logs = result.batchLogs || [];
      if (logs.length > 500) {
        const trimmedLogs = logs.slice(-500);
        chrome.storage.local.set({ batchLogs: trimmedLogs });
        console.log(`[Background] 清理了 ${logs.length - 500} 条旧日志`);
      }
    });
  }
});
