/**
 * Content Script - 悬浮窗版本
 * 协调所有模块,提供悬浮球界面
 */

// 获取扩展版本号
const EXTENSION_VERSION = chrome.runtime.getManifest().version;
console.log(`[即梦批量助手 v${EXTENSION_VERSION}] Content Script 已加载`);

// 防止重复注入
if (window.__JIMENG_CONTENT_INSTALLED) {
  console.warn('❌ 即梦批量助手已存在,跳过重复注入');
  throw new Error('插件已注入,禁止重复加载');
}
window.__JIMENG_CONTENT_INSTALLED = true;

// 全局状态
let isRunning = false;
let currentTask = null;
let isPaused = false;

// ==================== 创建悬浮窗界面 ====================

function createFloatingPanel() {
  // 清理旧的悬浮球和面板
  const oldBall = document.getElementById('jimeng-floating-ball');
  const oldPanel = document.getElementById('jimeng-floating-panel');
  if (oldBall) oldBall.remove();
  if (oldPanel) oldPanel.remove();

  // 1. 创建悬浮球
  const ball = document.createElement('div');
  ball.id = 'jimeng-floating-ball';
  ball.className = 'jimeng-ball';
  ball.innerHTML = '🎨';
  ball.title = '即梦批量助手';
  document.body.appendChild(ball);

  // 2. 创建弹窗面板
  const panel = document.createElement('div');
  panel.id = 'jimeng-floating-panel';
  panel.className = 'jimeng-panel';
  panel.style.display = 'none';

  // 标题栏
  const header = createHeader();
  panel.appendChild(header);

  // 内容区
  const content = createContent();
  panel.appendChild(content);

  // 底部固定按钮
  const footer = createFooter();
  panel.appendChild(footer);

  document.body.appendChild(panel);

  // 点击悬浮球展开/收起
  ball.onclick = () => {
    if (panel.style.display === 'none') {
      panel.style.display = 'flex';
      loadHistoryLogs();
    } else {
      panel.style.display = 'none';
    }
  };

  // 使悬浮球可拖拽
  makeDraggable(ball);
}

// 创建标题栏
function createHeader() {
  const header = document.createElement('div');
  header.className = 'jimeng-header';

  // 获取扩展版本号
  const version = chrome.runtime.getManifest().version;

  header.innerHTML = `
    <div class="jimeng-header-title">
      <span class="jimeng-logo">🎨</span>
      <span class="jimeng-title-text">即梦批量助手</span>
      <span class="jimeng-version">v${version}</span>
    </div>
  `;

  const closeBtn = document.createElement('button');
  closeBtn.className = 'jimeng-close-btn';
  closeBtn.innerHTML = '✕';
  closeBtn.onclick = () => {
    document.getElementById('jimeng-floating-panel').style.display = 'none';
  };
  header.querySelector('.jimeng-header-title').appendChild(closeBtn);

  return header;
}

// 创建内容区
function createContent() {
  const content = document.createElement('div');
  content.className = 'jimeng-content';
  content.innerHTML = `
    <!-- 文件上传区 -->
    <div class="jimeng-section">
      <div class="jimeng-section-title">📂 分镜文件</div>
      <div class="jimeng-file-upload" id="jimeng-file-upload">
        <input type="file" id="jimeng-file-input" accept=".txt,.csv" style="display: none;">
        <div class="jimeng-upload-placeholder" id="jimeng-upload-placeholder">
          <p class="jimeng-upload-hint">点击或拖拽文件到此处</p>
          <p class="jimeng-upload-sub">支持 .txt, .csv 格式</p>
        </div>
        <div class="jimeng-file-info" id="jimeng-file-info" style="display: none;">
          <span class="jimeng-file-name" id="jimeng-file-name"></span>
          <span class="jimeng-file-count" id="jimeng-file-count"></span>
          <button class="jimeng-btn-small" id="jimeng-clear-file">清除</button>
        </div>
      </div>
    </div>

    <!-- 参考图上传 - Tab 版本 -->
    <div class="jimeng-section">
      <div class="jimeng-section-title">
        🖼️ 参考图
        <div class="jimeng-ref-actions">
          <button class="jimeng-btn-icon" id="jimeng-add-tab" title="添加新Tab">+</button>
          <button class="jimeng-btn-icon jimeng-btn-clear-all" id="jimeng-clear-all-tabs" title="清空所有Tab的图片">
            <span style="font-size: 12px;">🗑️</span>
          </button>
        </div>
      </div>

      <!-- Tab 切换栏 -->
      <div class="jimeng-tabs" id="jimeng-tabs"></div>

      <!-- Tab 内容区 -->
      <div class="jimeng-tab-content">
        <!-- 图片上传按钮 -->
        <div class="jimeng-ref-upload">
          <button class="jimeng-btn-secondary jimeng-btn-small" id="jimeng-add-images">📁 添加图片 (原图)</button>
          <button class="jimeng-btn-secondary jimeng-btn-small" id="jimeng-clear-tab">清空当前</button>
          <span class="jimeng-ref-count" id="jimeng-ref-count">当前: 0 张</span>
        </div>

        <!-- 图片预览容器 -->
        <div class="jimeng-ref-preview" id="jimeng-ref-preview" style="display: none;">
          <div class="jimeng-ref-preview-list" id="jimeng-ref-preview-list"></div>
        </div>
      </div>
    </div>

    <!-- 设置面板 -->
    <div class="jimeng-section">
      <div class="jimeng-section-title">⚙️ 设置</div>
      <div class="jimeng-settings">
        <div class="jimeng-setting-row">
          <label>范围:</label>
          <div class="jimeng-range-input">
            <span>第</span>
            <input type="number" id="jimeng-start" value="1" min="1" class="jimeng-input-small">
            <span>~</span>
            <input type="number" id="jimeng-end" value="999" min="1" class="jimeng-input-small">
            <span>个</span>
          </div>
        </div>
        <div class="jimeng-setting-row">
          <label>间隔:</label>
          <div>
            <input type="number" id="jimeng-delay" value="2" min="1" max="60" class="jimeng-input-small">
            <span>秒</span>
          </div>
        </div>
        <div class="jimeng-setting-row">
          <label>重试:</label>
          <div>
            <input type="number" id="jimeng-retry" value="1" min="0" max="5" class="jimeng-input-small">
            <span>次</span>
          </div>
        </div>
      </div>
    </div>

    <!-- 进度条 -->
    <div class="jimeng-section" id="jimeng-progress-section" style="display: none;">
      <div class="jimeng-section-title">📊 进度</div>
      <div class="jimeng-progress-container">
        <div class="jimeng-progress-bar" id="jimeng-progress-bar">
          <span id="jimeng-progress-text">0%</span>
        </div>
      </div>
      <div class="jimeng-progress-detail" id="jimeng-progress-detail">0/0</div>
    </div>

    <!-- 日志区域 -->
    <div class="jimeng-section">
      <div class="jimeng-section-title">
        📋 日志
        <button class="jimeng-btn-small" id="jimeng-clear-log">清空</button>
      </div>
      <div class="jimeng-log-container" id="jimeng-log-container">
        <div class="jimeng-log-entry">
          <span class="jimeng-log-time">[等待开始]</span>
          <span class="jimeng-log-msg">准备就绪</span>
        </div>
      </div>
    </div>
  `;

  // 绑定事件
  setTimeout(() => bindEvents(), 100);

  return content;
}

// 创建底部固定按钮区
function createFooter() {
  const footer = document.createElement('div');
  footer.className = 'jimeng-footer';
  footer.innerHTML = `
    <div class="jimeng-actions">
      <button class="jimeng-btn-primary" id="jimeng-start-btn">🚀 开始批量</button>
      <button class="jimeng-btn-secondary jimeng-btn-small" id="jimeng-pause-btn" disabled>⏸️ 暂停</button>
    </div>
  `;

  return footer;
}

// 绑定事件
function bindEvents() {
  // 文件上传区点击
  const fileUpload = document.getElementById('jimeng-file-upload');
  const fileInput = document.getElementById('jimeng-file-input');
  if (fileUpload && fileInput) {
    fileUpload.onclick = () => fileInput.click();
    fileInput.onchange = handleFileSelect;
  }

  // 清除文件
  const clearFileBtn = document.getElementById('jimeng-clear-file');
  if (clearFileBtn) {
    clearFileBtn.onclick = (e) => {
      e.stopPropagation();
      clearFile();
    };
  }

  // Tab 管理按钮
  const addTabBtn = document.getElementById('jimeng-add-tab');
  if (addTabBtn) {
    addTabBtn.onclick = handleAddTab;
  }

  const clearAllTabsBtn = document.getElementById('jimeng-clear-all-tabs');
  if (clearAllTabsBtn) {
    clearAllTabsBtn.onclick = handleClearAllTabs;
  }

  // 参考图操作
  const addImagesBtn = document.getElementById('jimeng-add-images');
  if (addImagesBtn) {
    addImagesBtn.onclick = handleAddImages; // 直接调用，不再使用input
  }

  const clearTabBtn = document.getElementById('jimeng-clear-tab');
  if (clearTabBtn) {
    clearTabBtn.onclick = handleClearCurrentTab;
  }

  // 开始按钮
  const startBtn = document.getElementById('jimeng-start-btn');
  if (startBtn) {
    startBtn.onclick = handleStart;
  }

  // 暂停按钮
  const pauseBtn = document.getElementById('jimeng-pause-btn');
  if (pauseBtn) {
    pauseBtn.onclick = handlePause;
  }

  // 清空日志
  const clearLogBtn = document.getElementById('jimeng-clear-log');
  if (clearLogBtn) {
    clearLogBtn.onclick = handleClearLog;
  }
}

// ==================== 事件处理函数 ====================

let fileContent = null;
let storyboardCount = 0;

// 初始化 Tab 管理器
async function initTabsManager() {
  try {
    console.log('[即梦批量助手] 正在初始化 Tab 管理器...');
    await referenceTabsManager.init();
    renderTabs();
    updateTabPreview();
    addLog('✅ 参考图 Tab 管理器已初始化', 'success');

    // 显示存储使用情况
    const stats = referenceTabsManager.getStats();
    let totalImages = 0;
    for (const tab in stats) {
      totalImages += stats[tab].count;
    }
    if (totalImages > 0) {
      addLog(`📊 已加载 ${totalImages} 张参考图 (原图质量)`, 'info');
    }
  } catch (error) {
    addLog(`❌ Tab 管理器初始化失败: ${error.message}`, 'error');
    console.error('[Tab Manager] 初始化错误:', error);
    console.error('[Tab Manager] 错误堆栈:', error.stack);

    // 提供恢复建议
    addLog('⚠️ 如果问题持续，请尝试：1) 刷新页面 2) 重新加载扩展', 'warning');
  }
}

// 渲染 Tabs
function renderTabs() {
  const tabsContainer = document.getElementById('jimeng-tabs');
  if (!tabsContainer) return;

  const tabNames = referenceTabsManager.getTabNames();
  const activeTab = referenceTabsManager.getActiveTabName();

  tabsContainer.innerHTML = '';

  tabNames.forEach(tabName => {
    const stats = referenceTabsManager.getStats()[tabName];
    const count = stats ? stats.count : 0;

    const tabItem = document.createElement('div');
    tabItem.className = `jimeng-tab-item ${tabName === activeTab ? 'active' : ''}`;
    tabItem.innerHTML = `
      <span class="jimeng-tab-name" title="${tabName}">${tabName}</span>
      <span class="jimeng-tab-count">(${count})</span>
      <button class="jimeng-tab-delete" title="删除 Tab「${tabName}」">×</button>
    `;

    // 点击切换 Tab
    tabItem.onclick = (e) => {
      if (e.target.className === 'jimeng-tab-delete') return;
      referenceTabsManager.setActiveTab(tabName).then(() => {
        renderTabs();
        updateTabPreview();
      }).catch(error => {
        addLog(`❌ 切换 Tab 失败: ${error.message}`, 'error');
      });
    };

    // 删除 Tab
    tabItem.querySelector('.jimeng-tab-delete').onclick = async (e) => {
      e.stopPropagation();

      // 更详细的确认提示
      const confirmMsg = count > 0
        ? `确定要删除 Tab「${tabName}」吗？\n\n该Tab包含 ${count} 张图片，删除后无法恢复。`
        : `确定要删除 Tab「${tabName}」吗？`;

      if (confirm(confirmMsg)) {
        try {
          await referenceTabsManager.deleteTab(tabName);
          renderTabs();
          updateTabPreview();
          addLog(`🗑️ 已删除 Tab: ${tabName}`, 'success');
        } catch (error) {
          addLog(`❌ 删除 Tab 失败: ${error.message}`, 'error');
        }
      }
    };

    tabsContainer.appendChild(tabItem);
  });
}

// 更新当前 Tab 的图片预览
function updateTabPreview() {
  const images = referenceTabsManager.getActiveTabImages();
  const count = images.length;
  const activeTabName = referenceTabsManager.getActiveTabName();

  // 更新计数
  document.getElementById('jimeng-ref-count').textContent = `当前: ${count} 张`;

  // 更新预览
  const previewContainer = document.getElementById('jimeng-ref-preview');
  const previewList = document.getElementById('jimeng-ref-preview-list');

  if (count > 0) {
    previewList.innerHTML = '';

    images.forEach((imageInfo, index) => {
      // 现在 imageInfo 是 { id, name, type, size, thumbnail, handleId }
      const imgWrapper = document.createElement('div');
      imgWrapper.className = 'jimeng-ref-preview-item';

      // 使用已保存的缩略图
      const thumbnailSrc = imageInfo.thumbnail || 'data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" width="100" height="100"><rect fill="%23ddd" width="100" height="100"/><text x="50%" y="50%" text-anchor="middle" dy=".3em" fill="%23999">无预览</text></svg>';

      imgWrapper.innerHTML = `
        <img src="${thumbnailSrc}" alt="${imageInfo.name}">
        <div class="jimeng-ref-preview-name">${imageInfo.name}</div>
        <button class="jimeng-ref-preview-remove" data-index="${index}">×</button>
      `;
      previewList.appendChild(imgWrapper);

      // 绑定删除按钮
      imgWrapper.querySelector('.jimeng-ref-preview-remove').onclick = async (evt) => {
        evt.stopPropagation();
        try {
          await referenceTabsManager.removeImageFromActiveTab(index);
          renderTabs();
          updateTabPreview();
          addLog(`🗑️ 已从 Tab "${activeTabName}" 删除图片`, 'success');
        } catch (error) {
          addLog(`❌ 删除图片失败: ${error.message}`, 'error');
        }
      };
    });

    previewContainer.style.display = 'block';
  } else {
    previewContainer.style.display = 'none';
  }
}

// Tab 事件处理函数
async function handleAddTab() {
  const tabName = prompt('请输入新 Tab 名称:');
  if (!tabName) return;

  try {
    await referenceTabsManager.addTab(tabName.trim());
    renderTabs();
    updateTabPreview();
    addLog(`✅ 已添加 Tab: ${tabName}`, 'success');
  } catch (error) {
    addLog(`❌ 添加 Tab 失败: ${error.message}`, 'error');
  }
}

async function handleAddImages() {
  console.log('[即梦批量助手] 打开文件选择器...');

  try {
    // 检查浏览器是否支持
    if (!window.showOpenFilePicker) {
      alert('您的浏览器不支持 File System Access API\n\n请使用最新版 Chrome 浏览器 (版本 86+)');
      addLog('❌ 浏览器不支持 File System Access API', 'error');
      return;
    }

    // 显示正在处理的提示
    addLog('📂 请选择图片文件...', 'info');

    // 使用 File System Access API 选择文件
    const fileHandlePairs = await filePicker.pickImages();

    if (fileHandlePairs.length === 0) {
      console.log('[即梦批量助手] 用户取消选择');
      return;
    }

    console.log(`[即梦批量助手] 用户选择了 ${fileHandlePairs.length} 个文件`);
    addLog(`⏳ 正在添加 ${fileHandlePairs.length} 张图片 (保存原图质量)...`, 'info');

    const count = await referenceTabsManager.addImagesToActiveTab(fileHandlePairs);
    const activeTabName = referenceTabsManager.getActiveTabName();

    renderTabs();
    updateTabPreview();
    addLog(`✅ 为 Tab "${activeTabName}" 添加了 ${count} 张图片 (原图质量，已授权持久访问)`, 'success');

  } catch (error) {
    // 详细的错误日志
    console.error('[即梦批量助手] 添加图片失败:', error);
    console.error('[即梦批量助手] 错误堆栈:', error.stack);

    // 用户友好的错误提示
    let errorMsg = error.message;
    if (errorMsg.includes('not supported')) {
      errorMsg = '浏览器不支持 File System Access API，请使用最新版Chrome';
    } else if (errorMsg.includes('permission')) {
      errorMsg = '文件访问权限被拒绝，请重新选择';
    } else if (errorMsg.includes('IndexedDB')) {
      errorMsg = '存储失败，浏览器可能不支持IndexedDB';
    }

    addLog(`❌ 添加图片失败: ${errorMsg}`, 'error');
  }
}

async function handleClearCurrentTab() {
  const activeTabName = referenceTabsManager.getActiveTabName();
  const count = referenceTabsManager.getActiveTabImageCount();

  if (count === 0) {
    addLog('当前 Tab 没有图片', 'info');
    return;
  }

  if (!confirm(`确定要清空 Tab "${activeTabName}" 的 ${count} 张图片吗？`)) {
    return;
  }

  try {
    await referenceTabsManager.clearActiveTab();
    renderTabs();
    updateTabPreview();
    addLog(`✅ 已清空 Tab "${activeTabName}" 的 ${count} 张图片`, 'success');
  } catch (error) {
    addLog(`❌ 清空失败: ${error.message}`, 'error');
  }
}

async function handleClearAllTabs() {
  const stats = referenceTabsManager.getStats();
  let totalCount = 0;
  const tabList = [];

  for (const tab in stats) {
    totalCount += stats[tab].count;
    if (stats[tab].count > 0) {
      tabList.push(`  • ${tab}: ${stats[tab].count} 张`);
    }
  }

  if (totalCount === 0) {
    addLog('所有 Tab 都没有图片', 'info');
    return;
  }

  // 更详细的确认提示
  const confirmMsg = `确定要清空所有 Tabs 的图片吗？\n\n将清空以下内容：\n${tabList.join('\n')}\n\n共 ${totalCount} 张图片，删除后无法恢复。`;

  if (!confirm(confirmMsg)) {
    return;
  }

  try {
    await referenceTabsManager.clearAllTabs();
    renderTabs();
    updateTabPreview();
    addLog(`✅ 已清空所有 Tabs 的 ${totalCount} 张图片`, 'success');
  } catch (error) {
    addLog(`❌ 清空失败: ${error.message}`, 'error');
  }
}

let referenceFiles = []; // 保留用于兼容性

// 处理文件选择
async function handleFileSelect(e) {
  const file = e.target.files[0];
  if (!file) return;

  try {
    fileContent = await file.text();

    // 实际解析分镜内容，而不是估算
    const parsedStoryboards = storyboardParser.parse(fileContent);
    storyboardCount = parsedStoryboards.length;

    // 更新UI
    document.getElementById('jimeng-file-name').textContent = file.name;
    document.getElementById('jimeng-file-count').textContent = `(${storyboardCount} 个分镜)`;
    document.getElementById('jimeng-upload-placeholder').style.display = 'none';
    document.getElementById('jimeng-file-info').style.display = 'flex';
    document.getElementById('jimeng-end').value = storyboardCount;

    addLog(`✅ 已加载: ${file.name}, 解析到 ${storyboardCount} 个分镜`, 'success');
  } catch (error) {
    addLog(`❌ 文件读取失败: ${error.message}`, 'error');
  }
}

// 清除文件
function clearFile() {
  fileContent = null;
  storyboardCount = 0;
  document.getElementById('jimeng-file-input').value = '';
  document.getElementById('jimeng-upload-placeholder').style.display = 'block';
  document.getElementById('jimeng-file-info').style.display = 'none';
  addLog('已清除文件');
}

// 开始批量任务
async function handleStart() {
  if (!fileContent) {
    alert('请先选择分镜文件');
    return;
  }

  if (isRunning) {
    alert('任务正在执行中...');
    return;
  }

  try {
    // 获取预期的参考图数量
    const expectedCount = referenceTabsManager.getActiveTabImageCount();

    if (expectedCount > 0) {
      // 先请求权限（利用用户点击的手势上下文，会自动弹出系统权限对话框）
      addLog(`🔐 正在验证文件访问权限...`, 'info');
      await referenceTabsManager.requestPermissionsForActiveTab();
    }

    // 从当前 Tab 获取参考图（原图，自动验证权限）
    addLog(`⏳ 正在读取参考图 (预期 ${expectedCount} 张)...`, 'info');
    const referenceFiles = await referenceTabsManager.getActiveTabFiles();

    if (referenceFiles.length === 0) {
      if (expectedCount > 0) {
        // 有预期但实际读取为 0，可能是权限问题
        addLog(`⚠️ 读取参考图失败！预期 ${expectedCount} 张，实际读取 0 张`, 'warning');
        addLog('💡 提示: 可能是文件访问权限失效，请尝试重新添加参考图', 'warning');

        // 询问用户是否继续
        if (!confirm(`参考图读取失败（预期 ${expectedCount} 张，实际 0 张）\n\n这可能是因为文件访问权限失效。\n\n是否仍要继续？（不带参考图生成可能效果不佳）`)) {
          addLog('👤 用户取消任务', 'info');
          return;
        }

        // 询问是否要重新选择参考图
        if (confirm('是否现在重新选择参考图？\n\n点击"确定"将打开文件选择器')) {
          addLog('📂 请选择参考图文件...', 'info');
          await reauthorizeReferenceImages();
          // 重新开始流程
          handleStart();
          return;
        }
      } else {
        addLog('⚠️ 当前 Tab 没有参考图', 'warning');
      }
    } else if (referenceFiles.length < expectedCount) {
      // 部分读取成功
      addLog(`⚠️ 部分参考图读取失败: 成功 ${referenceFiles.length}/${expectedCount} 张`, 'warning');
      addLog('💡 部分文件可能已被移动或权限失效', 'info');
    } else {
      addLog(`✅ 成功读取 ${referenceFiles.length} 张参考图 (原图质量)`, 'success');
    }

    // 加载到 imageUploader
    await imageUploader.loadReferenceImages(referenceFiles);

    // 获取设置
    const settings = {
      startIndex: parseInt(document.getElementById('jimeng-start').value),
      endIndex: parseInt(document.getElementById('jimeng-end').value),
      retryTimes: parseInt(document.getElementById('jimeng-retry').value),
      delay: parseInt(document.getElementById('jimeng-delay').value) * 1000
    };

    // 开始执行
    startBatchTask(fileContent, settings);
  } catch (error) {
    console.error('[即梦批量助手] 获取参考图失败:', error);
    addLog(`❌ 获取参考图失败: ${error.message}`, 'error');

    // 如果是权限问题，提供友好提示
    if (error.message.includes('permission') || error.message.includes('权限')) {
      alert('无法读取参考图，可能是文件访问权限失效。\n\n请尝试：\n1. 重新添加参考图\n2. 确保图片文件未被移动或删除');
    }
  }
}

// 重新授权参考图 - 让用户重新选择文件
async function reauthorizeReferenceImages() {
  try {
    // 清空当前 Tab 的图片
    await referenceTabsManager.clearActiveTab();

    // 打开文件选择器
    const fileHandles = await window.showOpenFilePicker({
      multiple: true,
      types: [{
        description: '图片文件',
        accept: { 'image/*': ['.png', '.jpg', '.jpeg', '.webp', '.gif'] }
      }]
    });

    if (fileHandles.length === 0) {
      addLog('⚠️ 未选择任何文件', 'warning');
      return;
    }

    // 读取文件
    const fileHandlePairs = [];
    for (const handle of fileHandles) {
      const file = await handle.getFile();
      fileHandlePairs.push({ handle, file });
    }

    // 添加到当前 Tab
    await referenceTabsManager.addImagesToActiveTab(fileHandlePairs);

    // 更新 UI
    renderTabs();
    updateTabPreview();

    addLog(`✅ 已重新添加 ${fileHandles.length} 张参考图`, 'success');

  } catch (error) {
    if (error.name === 'AbortError') {
      addLog('👤 用户取消了文件选择', 'info');
    } else {
      console.error('[即梦批量助手] 重新授权失败:', error);
      addLog(`❌ 重新授权失败: ${error.message}`, 'error');
    }
  }
}

// 暂停任务
function handlePause() {
  isPaused = !isPaused;
  const pauseBtn = document.getElementById('jimeng-pause-btn');
  if (isPaused) {
    pauseBtn.textContent = '▶️ 继续';
    addLog('任务已暂停');
  } else {
    pauseBtn.textContent = '⏸️ 暂停';
    addLog('任务继续执行');
  }
}

// 清空日志
function handleClearLog() {
  const logContainer = document.getElementById('jimeng-log-container');
  if (logContainer) {
    logContainer.innerHTML = '<div class="jimeng-log-entry"><span class="jimeng-log-time">[已清空]</span><span class="jimeng-log-msg">日志已清空</span></div>';
  }
  chrome.storage.local.set({ batchLogs: [] });
}

// ==================== 批量处理逻辑 ====================

async function startBatchTask(content, settings) {
  console.log('[即梦批量助手] 开始批量任务');
  isRunning = true;

  // 更新UI
  document.getElementById('jimeng-start-btn').disabled = true;
  document.getElementById('jimeng-pause-btn').disabled = false;
  document.getElementById('jimeng-progress-section').style.display = 'block';

  try {
    // 解析分镜
    const storyboards = storyboardParser.parse(content);
    if (storyboards.length === 0) {
      addLog('❌ 未解析到任何分镜', 'error');
      return;
    }

    addLog(`📝 解析到 ${storyboards.length} 个分镜`);

    // 批量处理
    const result = await batchProcess(storyboards, settings);

    // 完成
    addLog(`✅ 批量任务完成! 成功: ${result.success}, 失败: ${result.failed}`, 'success');
    if (result.failedIndices.length > 0) {
      addLog(`失败的分镜: ${result.failedIndices.join(', ')}`, 'warning');
    }

  } catch (error) {
    console.error('[即梦批量助手] 任务异常:', error);
    addLog(`❌ 任务异常: ${error.message}`, 'error');
  } finally {
    isRunning = false;
    isPaused = false;
    document.getElementById('jimeng-start-btn').disabled = false;
    document.getElementById('jimeng-pause-btn').disabled = true;
  }
}

async function batchProcess(storyboards, settings) {
  const { startIndex = 1, endIndex, retryTimes = 2, delay = 3000 } = settings;
  const actualEndIndex = endIndex || storyboards.length;

  const results = {
    total: actualEndIndex - startIndex + 1,
    success: 0,
    failed: 0,
    failedIndices: []
  };

  // 配置提交处理器
  submitHandler.maxRetries = retryTimes;
  submitHandler.settings.submitDelay = delay;

  // 逐个处理
  for (let i = startIndex - 1; i < actualEndIndex; i++) {
    // 检查暂停
    while (isPaused) {
      await wait(500);
    }

    const storyboard = storyboards[i];
    const actualIndex = i + 1;

    // 更新进度
    updateProgress(actualIndex - startIndex + 1, results.total);

    // 处理单个分镜
    const success = await submitHandler.processOne(storyboard, actualIndex, actualEndIndex);

    if (success) {
      results.success++;
      addLog(`✅ #${actualIndex} 提交成功`, 'success');
    } else {
      results.failed++;
      results.failedIndices.push(actualIndex);
      addLog(`❌ #${actualIndex} 提交失败`, 'error');
    }
  }

  return results;
}

// ==================== 工具函数 ====================

function wait(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

function updateProgress(current, total) {
  const percent = Math.round((current / total) * 100);
  const progressBar = document.getElementById('jimeng-progress-bar');
  const progressText = document.getElementById('jimeng-progress-text');
  const progressDetail = document.getElementById('jimeng-progress-detail');

  if (progressBar) progressBar.style.width = percent + '%';
  if (progressText) progressText.textContent = percent + '%';
  if (progressDetail) progressDetail.textContent = `${current}/${total}`;
}

function addLog(message, type = 'info') {
  const logContainer = document.getElementById('jimeng-log-container');
  if (!logContainer) return;

  const time = new Date().toLocaleTimeString();
  const entry = document.createElement('div');
  entry.className = `jimeng-log-entry jimeng-log-${type}`;
  entry.innerHTML = `
    <span class="jimeng-log-time">[${time}]</span>
    <span class="jimeng-log-msg">${message}</span>
  `;

  logContainer.appendChild(entry);
  logContainer.scrollTop = logContainer.scrollHeight;

  // 保存到 storage
  chrome.storage.local.get(['batchLogs'], (res) => {
    const logs = res.batchLogs || [];
    logs.push({ time, message, type });
    if (logs.length > 50) logs.shift();
    chrome.storage.local.set({ batchLogs: logs });
  });
}

function loadHistoryLogs() {
  chrome.storage.local.get(['batchLogs'], (res) => {
    const logs = res.batchLogs || [];
    const logContainer = document.getElementById('jimeng-log-container');
    if (!logContainer) return;

    logContainer.innerHTML = '';
    if (logs.length === 0) {
      logContainer.innerHTML = '<div class="jimeng-log-empty">暂无历史日志</div>';
      return;
    }

    logs.slice(-20).forEach(log => {
      const entry = document.createElement('div');
      entry.className = `jimeng-log-entry jimeng-log-${log.type || 'info'}`;
      entry.innerHTML = `
        <span class="jimeng-log-time">[${log.time}]</span>
        <span class="jimeng-log-msg">${log.message}</span>
      `;
      logContainer.appendChild(entry);
    });
    logContainer.scrollTop = logContainer.scrollHeight;
  });
}

// 使元素可拖拽
function makeDraggable(element) {
  let isDragging = false;
  let currentX;
  let currentY;
  let initialX;
  let initialY;

  element.addEventListener('mousedown', dragStart);
  document.addEventListener('mousemove', drag);
  document.addEventListener('mouseup', dragEnd);

  function dragStart(e) {
    initialX = e.clientX - element.offsetLeft;
    initialY = e.clientY - element.offsetTop;
    isDragging = true;
  }

  function drag(e) {
    if (isDragging) {
      e.preventDefault();
      currentX = e.clientX - initialX;
      currentY = e.clientY - initialY;
      element.style.right = 'auto';
      element.style.bottom = 'auto';
      element.style.left = currentX + 'px';
      element.style.top = currentY + 'px';
    }
  }

  function dragEnd() {
    isDragging = false;
  }
}

// ==================== 初始化 ====================

// 智能初始化函数
function initFloatingPanel() {
  // 防止重复创建
  if (document.getElementById('jimeng-floating-ball')) {
    console.log(`[即梦批量助手 v${EXTENSION_VERSION}] 悬浮球已存在,跳过创建`);
    return;
  }

  // 检查页面是否准备就绪
  if (document.body) {
    createFloatingPanel();
    console.log(`[即梦批量助手 v${EXTENSION_VERSION}] ✅ 悬浮窗创建完成`);

    // 立即初始化Tab管理器（页面加载时就初始化，确保刷新后图片不丢失）
    initTabsManager();
  } else {
    console.warn(`[即梦批量助手 v${EXTENSION_VERSION}] ⚠️ document.body 未就绪,等待重试...`);
    setTimeout(initFloatingPanel, 100);
  }
}

// ==================== 全局调试工具 ====================
// 在控制台可以使用这些函数进行诊断和修复

window.jimengDebug = {
  // 诊断Tab管理器
  async diagnose() {
    console.log('🔍 开始诊断 Tab 管理器...');
    await referenceTabsManager.diagnoseAndFix();
    console.log('✅ 诊断完成，请查看上方日志');
  },

  // 重置所有数据
  async reset() {
    if (!confirm('⚠️ 确定要重置所有参考图数据吗？这将清除所有已保存的图片！')) {
      return;
    }
    console.log('🔄 开始重置...');
    await referenceTabsManager.reset();
    console.log('✅ 重置完成，请刷新页面');
    alert('重置完成！请刷新页面。');
  },

  // 查看当前状态
  status() {
    console.log('📊 Tab 管理器状态:');
    console.log('  activeTab:', referenceTabsManager.getActiveTabName());
    console.log('  tabs:', referenceTabsManager.getTabNames());
    console.log('  stats:', referenceTabsManager.getStats());
  },

  // 查看IndexedDB统计
  async storage() {
    const stats = await indexedDBStorage.getStorageStats();
    console.log('💾 IndexedDB 存储统计:');
    console.log('  总图片数:', stats.itemCount);
    console.log('  总大小:', stats.totalSizeMB, 'MB');
  },

  // 帮助信息
  help() {
    console.log(`
🛠️  即梦批量助手 - 调试工具
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
可用命令：

  jimengDebug.diagnose()  - 诊断并修复Tab管理器问题
  jimengDebug.reset()     - 重置所有数据（清空图片）
  jimengDebug.status()    - 查看当前状态
  jimengDebug.storage()   - 查看存储统计
  jimengDebug.help()      - 显示此帮助信息

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
如果添加图片失败，请尝试：
1. jimengDebug.diagnose() - 尝试自动修复
2. 刷新页面
3. jimengDebug.reset() - 清空所有数据重新开始
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
    `);
  }
};

// 初始化时显示帮助提示
console.log(`
🎨 即梦批量助手 v${EXTENSION_VERSION} 已加载
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
遇到问题？在控制台输入: jimengDebug.help()
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
`);

// ==================== 外部消息监听（接收 You 网站发来的数据）====================
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[即梦批量助手] 收到消息:', message);

  if (message.action === 'IMPORT_CSV_FROM_EXTERNAL') {
    console.log('[即梦批量助手] 接收到外部导入的 CSV 数据');

    // 使用 async 函数处理
    (async () => {
      try {
        console.log('[即梦批量助手] 开始处理外部导入消息...');

        // 首先展开面板（在处理其他内容之前）
        setTimeout(() => {
          const panel = document.getElementById('jimeng-floating-panel');
          console.log('[即梦批量助手] 尝试展开面板, panel存在:', !!panel);
          if (panel && panel.style.display === 'none') {
            panel.style.display = 'flex';
            console.log('[即梦批量助手] ✅ 已展开面板');
          }
        }, 2000);

        fileContent = message.csvData;

        // 解析分镜内容
        const parsedStoryboards = storyboardParser.parse(fileContent);
        storyboardCount = parsedStoryboards.length;

        // 更新 UI（添加错误处理）
        const fileNameEl = document.getElementById('jimeng-file-name');
        const fileCountEl = document.getElementById('jimeng-file-count');
        const uploadPlaceholder = document.getElementById('jimeng-upload-placeholder');
        const fileInfo = document.getElementById('jimeng-file-info');
        const endInput = document.getElementById('jimeng-end');

        if (fileNameEl) fileNameEl.textContent = message.fileName || '外部导入.csv';
        if (fileCountEl) fileCountEl.textContent = `(${storyboardCount} 个分镜)`;
        if (uploadPlaceholder) uploadPlaceholder.style.display = 'none';
        if (fileInfo) fileInfo.style.display = 'flex';
        if (endInput) endInput.value = storyboardCount;

        addLog(`✅ 外部导入成功: ${message.fileName || '外部导入.csv'}, ${storyboardCount} 个分镜`, 'success');

        // 处理角色图片（如果有）
        const characterImages = message.characterImages || [];
        if (characterImages.length > 0) {
          await importCharacterImages(characterImages);
        }

        // 延迟展开悬浮窗（通过直接修改 style）
        const expandPanel = () => {
          const panel = document.getElementById('jimeng-floating-panel');
          console.log('[即梦批量助手] expandPanel 被调用, panel存在:', !!panel, 'display:', panel?.style?.display);
          if (panel && panel.style.display === 'none') {
            panel.style.display = 'flex';
            console.log('[即梦批量助手] ✅ 已展开面板');
            return true;
          } else if (panel && panel.style.display === 'flex') {
            console.log('[即梦批量助手] 面板已是展开状态');
            return true;
          }
          return false;
        };

        // 等待更长时间再尝试展开（等待 initFloatingPanel 完成）
        console.log('[即梦批量助手] 开始等待面板初始化...');
        setTimeout(() => {
          console.log('[即梦批量助手] 第一次尝试展开面板');
          if (!expandPanel()) {
            setTimeout(() => {
              console.log('[即梦批量助手] 第二次尝试展开面板');
              if (!expandPanel()) {
                setTimeout(() => {
                  console.log('[即梦批量助手] 第三次尝试展开面板');
                  expandPanel();
                }, 3000);
              }
            }, 2000);
          }
        }, 2000);

        // 自动开始生成（延迟5秒确保面板已展开）
        setTimeout(() => {
          const startBtn = document.getElementById('jimeng-start-btn');
          if (startBtn && !startBtn.disabled) {
            addLog('🚀 自动开始批量生成...', 'info');
            startBtn.click();
          }
        }, 5000);

        sendResponse({ success: true, count: storyboardCount });
      } catch (error) {
        console.error('[即梦批量助手] 外部导入失败:', error);
        addLog(`❌ 外部导入失败: ${error.message}`, 'error');
        sendResponse({ success: false, error: error.message });
      }
    })();

    return true;  // 异步响应
  }
});

// 导入角色图片到参考图（按分类创建不同标签）
async function importCharacterImages(images) {
  if (!images || images.length === 0) return;

  try {
    addLog(`⏳ 正在导入 ${images.length} 张角色图片...`, 'info');

    // 按分类分组图片
    const imagesByCategory = {};
    for (const img of images) {
      const categoryName = img.categoryName || '默认';
      if (!imagesByCategory[categoryName]) {
        imagesByCategory[categoryName] = [];
      }
      imagesByCategory[categoryName].push(img);
    }

    // 处理每个分类
    for (const [categoryName, categoryImages] of Object.entries(imagesByCategory)) {
      const tabName = categoryName;

      // 尝试找到或创建对应 Tab
      const existingTabs = referenceTabsManager.getTabNames();
      if (!existingTabs.includes(tabName)) {
        await referenceTabsManager.addTab(tabName);
      }
      await referenceTabsManager.setActiveTab(tabName);
      await referenceTabsManager.clearActiveTab();

      // 确保 tab 数组存在
      if (!referenceTabsManager.tabs[tabName]) {
        referenceTabsManager.tabs[tabName] = [];
      }

      // 获取现有缩略图数据
      const thumbnailsKey = 'jimeng_reference_thumbnails';
      const storageResult = await chrome.storage.local.get(thumbnailsKey);
      const thumbnails = storageResult[thumbnailsKey] || {};

      // 将图片添加到该 Tab
      let addedCount = 0;
      for (let i = 0; i < categoryImages.length; i++) {
        const img = categoryImages[i];
        try {
          const imageId = 'ext_' + Date.now() + '_' + Math.random().toString(36).substr(2, 6);

          // 生成缩略图
          const thumbnail = await generateThumbnail(img.imageData);

          // 保存缩略图到 storage
          thumbnails[imageId] = thumbnail;

          // 添加到内存中的 tab
          referenceTabsManager.tabs[tabName].push({
            id: imageId,
            name: img.name,
            type: img.imageType || 'image/webp',
            size: img.imageData.length,
            thumbnail: thumbnail,
            fullData: img.imageData,  // 保存完整图片数据
            handleId: null
          });

          addedCount++;
        } catch (err) {
          console.error(`[即梦批量助手] 处理角色图片失败: ${img.name}`, err);
        }
      }

      // 保存缩略图到 chrome.storage
      await chrome.storage.local.set({ [thumbnailsKey]: thumbnails });

      addLog(`✅ 已导入 ${addedCount} 张图片到 "${tabName}" Tab`, 'success');
    }

    renderTabs();
    updateTabPreview();

  } catch (error) {
    console.error('[即梦批量助手] 导入角色图片失败:', error);
    addLog(`❌ 导入角色图片失败: ${error.message}`, 'error');
  }
}

// 辅助函数：读取文件为 DataURL
function readFileAsDataURL(file) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onload = () => resolve(reader.result);
    reader.onerror = reject;
    reader.readAsDataURL(file);
  });
}

// 辅助函数：生成缩略图
function generateThumbnail(dataUrl, maxSize = 100) {
  return new Promise((resolve) => {
    const img = new Image();
    img.onload = () => {
      const canvas = document.createElement('canvas');
      const scale = Math.min(maxSize / img.width, maxSize / img.height);
      canvas.width = img.width * scale;
      canvas.height = img.height * scale;
      const ctx = canvas.getContext('2d');
      ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
      resolve(canvas.toDataURL('image/jpeg', 0.7));
    };
    img.onerror = () => resolve(dataUrl);
    img.src = dataUrl;
  });
}

// 检查是否有暂存的待导入数据
function checkPendingImport() {
  console.log('[即梦批量助手] 检查是否有暂存数据...');

  chrome.storage.local.get(['pendingCsvImport', 'pendingFileName', 'pendingCharacterImages'], (result) => {
    console.log('[即梦批量助手] storage 读取结果:', {
      hasCsvData: !!result.pendingCsvImport,
      fileName: result.pendingFileName,
      imageCount: result.pendingCharacterImages?.length || 0
    });

    if (result.pendingCsvImport) {
      console.log('[即梦批量助手] 发现暂存的导入数据，正在自动导入...');

      // 延迟执行，确保 UI 已初始化
      setTimeout(async () => {
        try {
          fileContent = result.pendingCsvImport;
          const parsedStoryboards = storyboardParser.parse(fileContent);
          storyboardCount = parsedStoryboards.length;

          document.getElementById('jimeng-file-name').textContent = result.pendingFileName || '外部导入.csv';
          document.getElementById('jimeng-file-count').textContent = `(${storyboardCount} 个分镜)`;
          document.getElementById('jimeng-upload-placeholder').style.display = 'none';
          document.getElementById('jimeng-file-info').style.display = 'flex';
          document.getElementById('jimeng-end').value = storyboardCount;

          addLog(`✅ 自动导入成功: ${result.pendingFileName || '外部导入.csv'}, ${storyboardCount} 个分镜`, 'success');

          // 处理角色图片（如果有）
          const characterImages = result.pendingCharacterImages || [];
          if (characterImages.length > 0) {
            await importCharacterImages(characterImages);
          }

          // 清除暂存数据
          chrome.storage.local.remove(['pendingCsvImport', 'pendingFileName', 'pendingCharacterImages']);

          // 展开悬浮窗（使用 style.display）
          const panel = document.getElementById('jimeng-floating-panel');
          if (panel) {
            panel.style.display = 'flex';
            console.log('[即梦批量助手] ✅ 已展开面板 (checkPendingImport)');
          }

          // 自动开始生成（延迟2秒确保UI准备好）
          setTimeout(() => {
            const startBtn = document.getElementById('jimeng-start-btn');
            if (startBtn && !startBtn.disabled) {
              addLog('🚀 自动开始批量生成...', 'info');
              startBtn.click();
            }
          }, 2000);
        } catch (error) {
          console.error('[即梦批量助手] 自动导入失败:', error);
          addLog(`❌ 自动导入失败: ${error.message}`, 'error');
        }
      }, 1000);
    }
  });
}

// 多种方式确保初始化
// 1. 如果 DOM 已就绪，立即初始化
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    console.log(`[即梦批量助手 v${EXTENSION_VERSION}] DOMContentLoaded 触发`);
    setTimeout(initFloatingPanel, 500);
    setTimeout(checkPendingImport, 3000);  // 增加延迟确保页面加载完成
  });
} else {
  // DOM 已经就绪，立即初始化
  console.log(`[即梦批量助手 v${EXTENSION_VERSION}] DOM 已就绪,立即初始化`);
  setTimeout(initFloatingPanel, 500);
  setTimeout(checkPendingImport, 3000);  // 增加延迟确保页面加载完成
}

// 2. window.load 事件作为备用
window.addEventListener('load', () => {
  console.log(`[即梦批量助手 v${EXTENSION_VERSION}] window.load 触发`);
  setTimeout(initFloatingPanel, 500);
  setTimeout(checkPendingImport, 4000);  // window.load 后也检查待导入数据
});

// 3. 使用 MutationObserver 监测页面内容加载完成
const observer = new MutationObserver((mutations, obs) => {
  // 如果检测到页面有足够的内容，尝试初始化
  if (document.body && document.body.children.length > 5) {
    console.log(`[即梦批量助手 v${EXTENSION_VERSION}] MutationObserver 检测到页面内容加载`);
    setTimeout(initFloatingPanel, 500);
    obs.disconnect(); // 停止观察
  }
});

// 开始观察
if (document.body) {
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
} else {
  // 如果 body 还未就绪，等待后再观察
  setTimeout(() => {
    if (document.body) {
      observer.observe(document.body, {
        childList: true,
        subtree: true
      });
    }
  }, 100);
}

