/**
 * DOM 操作辅助模块
 * 提供通用的 DOM 查找和操作方法
 */

class DOMHelper {
  /**
   * 通用元素查找器
   * @param {Array<string>} selectorList - 选择器列表
   * @param {Array<Function>} validators - 验证函数列表
   * @returns {Element|null} 找到的元素
   */
  findElement(selectorList, validators = []) {
    for (const selector of selectorList) {
      const elements = document.querySelectorAll(selector);

      for (const el of elements) {
        // 检查可见性
        if (el.offsetParent === null) continue;

        // 自定义验证
        const valid = validators.every(validator => validator(el));
        if (valid) return el;
      }
    }
    return null;
  }

  /**
   * 查找提交按钮
   * @returns {Element|null}
   */
  findSubmitButton() {
    const selectors = [
      "button.submit-button-VW0U_J",
      "button.lv-btn-primary",
      ".lv-btn.lv-btn-primary.submit-button-VW0U_J",
      "button[type='submit']"
    ];

    return this.findElement(
      selectors,
      [
        (el) => !el.disabled, // 未禁用
        (el) => /生成|创作|提交/.test(el.textContent), // 文本匹配
        (el) => el.offsetParent !== null // 可见
      ]
    );
  }

  /**
   * 查找输入框
   * @returns {Element|null}
   */
  findTextarea() {
    const selectors = [
      "textarea[placeholder*='图片']",
      "textarea[placeholder*='提示']",
      "textarea.prompt-textarea-XfqAoB",
      "textarea"
    ];

    return this.findElement(
      selectors,
      [
        (el) => el.offsetParent !== null, // 可见
        (el) => !el.disabled // 未禁用
      ]
    );
  }

  /**
   * 等待元素出现
   * @param {string} selector - 选择器
   * @param {number} timeout - 超时时间(ms)
   * @returns {Promise<Element>}
   */
  waitForElement(selector, timeout = 5000) {
    return new Promise((resolve, reject) => {
      const element = document.querySelector(selector);
      if (element) {
        resolve(element);
        return;
      }

      const observer = new MutationObserver(() => {
        const element = document.querySelector(selector);
        if (element) {
          observer.disconnect();
          clearTimeout(timeoutId);
          resolve(element);
        }
      });

      observer.observe(document.body, {
        childList: true,
        subtree: true
      });

      const timeoutId = setTimeout(() => {
        observer.disconnect();
        reject(new Error(`元素 ${selector} 未在 ${timeout}ms 内出现`));
      }, timeout);
    });
  }

  /**
   * 模拟人类点击
   * @param {Element} element - 要点击的元素
   */
  async humanClick(element) {
    // 先 hover
    element.dispatchEvent(new MouseEvent('mouseover', { bubbles: true }));
    await this.delay(100 + Math.random() * 200);

    // 移动到元素
    element.scrollIntoView({ behavior: 'smooth', block: 'center' });
    await this.delay(200 + Math.random() * 300);

    // 点击
    const event = new MouseEvent('click', {
      bubbles: true,
      cancelable: true,
      view: window
    });
    element.dispatchEvent(event);
  }

  /**
   * 延迟函数
   * @param {number} ms - 延迟时间(ms)
   */
  delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }

  /**
   * 随机延迟
   * @param {number} min - 最小时间(ms)
   * @param {number} max - 最大时间(ms)
   */
  randomDelay(min, max) {
    const delay = Math.random() * (max - min) + min;
    return this.delay(delay);
  }
}

// 导出单例
const domHelper = new DOMHelper();
