/**
 * File Handle 存储模块
 * 使用 File System Access API 存储文件句柄，实现权限持久化
 * 关键：FileSystemFileHandle 可以存储在IndexedDB中，并保留权限
 */

class FileHandleStorage {
  constructor() {
    this.dbName = 'JimengFileHandles';
    this.version = 1;
    this.storeName = 'handles';
    this.db = null;
  }

  /**
   * 初始化数据库
   */
  async init() {
    return new Promise((resolve, reject) => {
      const request = indexedDB.open(this.dbName, this.version);

      request.onerror = () => {
        console.error('[FileHandleStorage] 打开失败:', request.error);
        reject(request.error);
      };

      request.onsuccess = () => {
        this.db = request.result;
        console.log('[FileHandleStorage] ✅ 数据库已打开');
        resolve();
      };

      request.onupgradeneeded = (event) => {
        const db = event.target.result;

        // 创建对象存储
        if (!db.objectStoreNames.contains(this.storeName)) {
          const objectStore = db.createObjectStore(this.storeName, { keyPath: 'id' });
          objectStore.createIndex('tabName', 'tabName', { unique: false });
          objectStore.createIndex('fileName', 'fileName', { unique: false });
          console.log('[FileHandleStorage] 对象存储已创建');
        }
      };
    });
  }

  /**
   * 保存文件句柄
   * @param {string} tabName - Tab名称
   * @param {Array<{handle: FileSystemFileHandle, file: File}>} fileHandlePairs - 文件句柄和文件对象的配对列表
   * @returns {Array} 保存的文件信息
   */
  async saveFileHandles(tabName, fileHandlePairs) {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readwrite');
    const objectStore = transaction.objectStore(this.storeName);

    const savedFiles = [];

    for (const pair of fileHandlePairs) {
      try {
        const { handle, file } = pair;

        const id = `${tabName}_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

        const handleData = {
          id: id,
          tabName: tabName,
          fileName: file.name,
          fileType: file.type,
          fileSize: file.size,
          handle: handle, // FileSystemFileHandle 可以直接存储在IndexedDB中
          timestamp: Date.now()
        };

        await this.promisifyRequest(objectStore.put(handleData));

        savedFiles.push({
          id: id,
          name: file.name,
          type: file.type,
          size: file.size
        });

        console.log(`[FileHandleStorage] ✅ 保存文件句柄: ${file.name} (${(file.size / 1024).toFixed(1)}KB)`);
      } catch (error) {
        console.error(`[FileHandleStorage] ❌ 保存失败:`, error);
      }
    }

    return savedFiles;
  }

  /**
   * 获取指定Tab的所有文件（从handle读取）
   * @param {string} tabName - Tab名称
   * @returns {Array<Object>} 文件信息和handle列表
   */
  async getFileHandles(tabName) {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readonly');
    const objectStore = transaction.objectStore(this.storeName);
    const index = objectStore.index('tabName');

    const request = index.getAll(tabName);
    const results = await this.promisifyRequest(request);

    console.log(`[FileHandleStorage] 加载 Tab "${tabName}": ${results.length} 个文件句柄`);
    return results;
  }

  /**
   * 从handle读取文件，并验证权限
   * @param {FileSystemFileHandle} handle - 文件句柄
   * @param {Object} handleData - 可选，包含备份信息
   * @returns {File|null} 文件对象
   */
  async readFileFromHandle(handle, handleData = null) {
    try {
      if (!handle) {
        console.warn('[FileHandleStorage] ⚠️ 无效的文件句柄');
        return null;
      }

      // 检查权限
      let permission;
      try {
        permission = await handle.queryPermission({ mode: 'read' });
      } catch (queryError) {
        console.warn('[FileHandleStorage] ⚠️ 无法查询权限:', queryError.message);
        return null;
      }

      if (permission === 'granted') {
        // 权限已授予，直接读取
        try {
          const file = await handle.getFile();
          console.log(`[FileHandleStorage] ✅ 读取文件成功: ${file.name}`);
          return file;
        } catch (readError) {
          console.error('[FileHandleStorage] ❌ 读取文件失败:', readError.message);
          return null;
        }
      } else if (permission === 'prompt') {
        // 需要请求权限 - 这需要用户交互，可能会失败
        console.log('[FileHandleStorage] 📢 需要请求文件访问权限...');
        try {
          const newPermission = await handle.requestPermission({ mode: 'read' });
          if (newPermission === 'granted') {
            const file = await handle.getFile();
            console.log(`[FileHandleStorage] ✅ 权限授予后读取成功: ${file.name}`);
            return file;
          } else {
            console.warn('[FileHandleStorage] ⚠️ 用户拒绝了权限请求');
            return null;
          }
        } catch (requestError) {
          // 可能是因为不在用户手势上下文中
          console.warn('[FileHandleStorage] ⚠️ 请求权限失败 (可能需要用户交互):', requestError.message);
          return null;
        }
      } else {
        // 权限被拒绝
        console.warn('[FileHandleStorage] ⚠️ 读取权限已被拒绝');
        return null;
      }
    } catch (error) {
      console.error('[FileHandleStorage] ❌ 读取文件异常:', error);
      return null;
    }
  }

  /**
   * 获取所有Tab的文件信息（不读取文件内容）
   * @returns {Object} { tabName: [{ id, name, type, size }] }
   */
  async getAllTabsInfo() {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readonly');
    const objectStore = transaction.objectStore(this.storeName);

    const request = objectStore.getAll();
    const results = await this.promisifyRequest(request);

    // 按Tab分组
    const tabsInfo = {};
    for (const item of results) {
      if (!tabsInfo[item.tabName]) {
        tabsInfo[item.tabName] = [];
      }
      tabsInfo[item.tabName].push({
        id: item.id,
        name: item.fileName,
        type: item.fileType,
        size: item.fileSize,
        hasHandle: !!item.handle
      });
    }

    return tabsInfo;
  }

  /**
   * 删除文件句柄
   * @param {string} handleId - 句柄ID
   */
  async deleteHandle(handleId) {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readwrite');
    const objectStore = transaction.objectStore(this.storeName);

    await this.promisifyRequest(objectStore.delete(handleId));
    console.log(`[FileHandleStorage] 删除句柄: ${handleId}`);
  }

  /**
   * 删除指定Tab的所有句柄
   * @param {string} tabName - Tab名称
   */
  async deleteTabHandles(tabName) {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readwrite');
    const objectStore = transaction.objectStore(this.storeName);
    const index = objectStore.index('tabName');

    const request = index.getAllKeys(tabName);
    const keys = await this.promisifyRequest(request);

    for (const key of keys) {
      await this.promisifyRequest(objectStore.delete(key));
    }

    console.log(`[FileHandleStorage] 删除 Tab "${tabName}": ${keys.length} 个句柄`);
  }

  /**
   * 清空所有数据
   */
  async clearAll() {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readwrite');
    const objectStore = transaction.objectStore(this.storeName);

    await this.promisifyRequest(objectStore.clear());
    console.log('[FileHandleStorage] 已清空所有句柄');
  }

  /**
   * 验证文件句柄是否仍然有效
   * @param {FileSystemFileHandle} handle - 文件句柄
   * @returns {boolean} 是否有效
   */
  async verifyHandle(handle) {
    try {
      await handle.queryPermission({ mode: 'read' });
      return true;
    } catch (error) {
      console.warn('[FileHandleStorage] 句柄已失效:', error);
      return false;
    }
  }

  /**
   * 将IndexedDB请求转为Promise
   */
  promisifyRequest(request) {
    return new Promise((resolve, reject) => {
      request.onsuccess = () => resolve(request.result);
      request.onerror = () => reject(request.error);
    });
  }
}

// 导出单例
const fileHandleStorage = new FileHandleStorage();
