/**
 * File Picker 封装
 * 使用 showOpenFilePicker API 选择文件并获取 FileSystemFileHandle
 */

class FilePicker {
  /**
   * 打开文件选择器并获取文件句柄
   * @param {Object} options - 选项
   * @returns {Promise<Array<{handle: FileSystemFileHandle, file: File}>>}
   */
  async pickImages(options = {}) {
    // 检查浏览器是否支持 File System Access API
    if (!window.showOpenFilePicker) {
      throw new Error('您的浏览器不支持 File System Access API，请使用最新版Chrome');
    }

    try {
      const pickerOptions = {
        types: [
          {
            description: '图片文件',
            accept: {
              'image/*': ['.png', '.jpg', '.jpeg', '.gif', '.webp', '.bmp']
            }
          }
        ],
        multiple: options.multiple !== false, // 默认允许多选
        excludeAcceptAllOption: false
      };

      // 打开文件选择器
      const handles = await window.showOpenFilePicker(pickerOptions);

      console.log(`[FilePicker] 用户选择了 ${handles.length} 个文件`);

      // 从每个handle读取文件
      const results = [];
      for (const handle of handles) {
        try {
          const file = await handle.getFile();
          results.push({
            handle: handle,
            file: file
          });
          console.log(`[FilePicker] ✅ 获取文件: ${file.name}`);
        } catch (error) {
          console.error(`[FilePicker] ❌ 读取文件失败:`, error);
        }
      }

      return results;
    } catch (error) {
      // 用户取消选择
      if (error.name === 'AbortError') {
        console.log('[FilePicker] 用户取消选择');
        return [];
      }
      throw error;
    }
  }

  /**
   * 选择单个图片
   * @returns {Promise<{handle: FileSystemFileHandle, file: File}|null>}
   */
  async pickSingleImage() {
    const results = await this.pickImages({ multiple: false });
    return results.length > 0 ? results[0] : null;
  }
}

// 导出单例
const filePicker = new FilePicker();
