/**
 * 图片批量下载模块 (新增功能)
 * 监听生成的图片并支持批量下载
 */

class ImageDownloader {
  constructor() {
    this.generatedImages = [];
    this.observer = null;
    this.isMonitoring = false;
  }

  /**
   * 开始监听图片生成
   */
  startMonitoring() {
    if (this.isMonitoring) {
      console.warn(`[ImageDownloader] 已在监听中`);
      return;
    }

    this.isMonitoring = true;
    this.generatedImages = [];

    console.log(`[ImageDownloader] 开始监听图片生成...`);

    // 监听新增的图片元素
    this.observer = new MutationObserver((mutations) => {
      mutations.forEach(mutation => {
        mutation.addedNodes.forEach(node => {
          if (node.nodeType !== 1) return;

          // 查找图片元素
          const images = node.tagName === 'IMG'
            ? [node]
            : node.querySelectorAll('img');

          images.forEach(img => {
            const src = img.src || img.dataset.src;

            // 过滤: 只收集即梦生成的图片
            if (src &&
                src.includes('jimeng') &&
                !this.generatedImages.some(item => item.src === src)) {

              const imageInfo = {
                src: src,
                alt: img.alt || '未命名',
                timestamp: Date.now(),
                element: img
              };

              this.generatedImages.push(imageInfo);
              console.log(`[ImageDownloader] 检测到新图片: ${src.substring(0, 50)}...`);

              // 通知 popup 更新计数
              this.notifyPopup();
            }
          });
        });
      });
    });

    this.observer.observe(document.body, {
      childList: true,
      subtree: true
    });
  }

  /**
   * 停止监听
   */
  stopMonitoring() {
    if (this.observer) {
      this.observer.disconnect();
      this.observer = null;
    }
    this.isMonitoring = false;
    console.log(`[ImageDownloader] 停止监听`);
  }

  /**
   * 批量下载图片 (逐个下载)
   * @param {Array} images - 图片列表 (可选,默认下载全部)
   */
  async downloadAll(images = this.generatedImages) {
    if (images.length === 0) {
      console.warn('[ImageDownloader] 没有检测到生成的图片');
      return;
    }

    console.log(`[ImageDownloader] 开始下载 ${images.length} 张图片...`);

    for (let i = 0; i < images.length; i++) {
      const img = images[i];
      const filename = `即梦图片_${i + 1}_${Date.now()}.png`;

      await this.downloadSingle(img.src, filename);

      // 延迟避免浏览器阻止多次下载
      await this.delay(800);
    }

    console.log(`[ImageDownloader] 下载完成`);
  }

  /**
   * 单张图片下载
   * @param {string} url - 图片 URL
   * @param {string} filename - 文件名
   */
  async downloadSingle(url, filename) {
    try {
      // 方法1: 使用 <a> 标签下载
      const a = document.createElement('a');
      a.href = url;
      a.download = filename;
      a.style.display = 'none';
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);

      console.log(`[ImageDownloader] 下载: ${filename}`);

    } catch (error) {
      console.error(`[ImageDownloader] 下载失败:`, error);

      // 方法2: 使用 chrome.downloads API (需通过 background.js)
      chrome.runtime.sendMessage({
        action: 'download',
        url: url,
        filename: filename
      });
    }
  }

  /**
   * 清空已检测的图片
   */
  clear() {
    this.generatedImages = [];
    console.log(`[ImageDownloader] 已清空图片列表`);
  }

  /**
   * 获取已检测的图片数量
   */
  getCount() {
    return this.generatedImages.length;
  }

  /**
   * 通知 popup 更新
   */
  notifyPopup() {
    try {
      chrome.runtime.sendMessage({
        action: 'updateImageCount',
        count: this.generatedImages.length
      });
    } catch (error) {
      // popup 可能未打开,忽略错误
    }
  }

  delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }
}

// 导出单例
const imageDownloader = new ImageDownloader();
