/**
 * 图片上传模块
 * 对应 Python 的 upload_reference_images() 函数
 */

class ImageUploader {
  constructor() {
    this.referenceImages = new Map(); // 角色名 -> File 对象
  }

  /**
   * 从用户选择的文件加载参考图
   * @param {FileList|Array<File>} files - 用户选择的文件列表
   */
  async loadReferenceImages(files) {
    this.referenceImages.clear();

    const fileArray = Array.isArray(files) ? files : Array.from(files);

    for (const file of fileArray) {
      // 验证文件类型
      if (!file.type.startsWith('image/')) {
        console.warn(`[ImageUploader] 跳过非图片文件: ${file.name}`);
        continue;
      }

      // 从文件名提取角色名 (女儿.png -> 女儿)
      const roleName = file.name.replace(/\.(png|jpg|jpeg|webp|bmp|gif)$/i, '');

      this.referenceImages.set(roleName, file);
      console.log(`[ImageUploader] 加载参考图: ${roleName} (${file.size} bytes)`);
    }

    console.log(`[ImageUploader] 共加载 ${this.referenceImages.size} 张参考图`);
  }

  /**
   * 根据角色名上传参考图
   * @param {Array<string>} roles - 角色名列表
   * @returns {Promise<boolean>} 是否成功
   */
  async uploadForRoles(roles) {
    if (roles.length === 0) {
      console.log(`[ImageUploader] 无需上传参考图`);
      return true;
    }

    const filesToUpload = [];
    const missingRoles = [];

    for (const role of roles) {
      if (this.referenceImages.has(role)) {
        filesToUpload.push(this.referenceImages.get(role));
      } else {
        missingRoles.push(role);
      }
    }

    if (missingRoles.length > 0) {
      console.warn(`[ImageUploader] 缺少参考图: ${missingRoles.join(', ')}`);
    }

    if (filesToUpload.length === 0) {
      console.log(`[ImageUploader] 没有可上传的参考图`);
      return true;
    }

    // 查找文件上传控件
    const fileInputSelectors = [
      'input[type="file"]',
      'input[accept*="image"]',
      'input.file-input-O6KAhP'
    ];

    let fileInput = null;
    for (const selector of fileInputSelectors) {
      const inputs = document.querySelectorAll(selector);
      for (const input of inputs) {
        // 检查是否在 DOM 中且可用
        if (input.isConnected && !input.disabled) {
          fileInput = input;
          break;
        }
      }
      if (fileInput) break;
    }

    if (!fileInput) {
      console.error(`[ImageUploader] 未找到文件上传控件`);
      return false;
    }

    try {
      // 使用 DataTransfer 设置文件
      const dataTransfer = new DataTransfer();
      filesToUpload.forEach(file => dataTransfer.items.add(file));

      fileInput.files = dataTransfer.files;

      // 触发 change 事件
      fileInput.dispatchEvent(new Event('change', { bubbles: true }));
      fileInput.dispatchEvent(new Event('input', { bubbles: true }));

      console.log(`[ImageUploader] 成功上传 ${filesToUpload.length} 张参考图`);

      // 等待上传处理
      await this.delay(2000);

      return true;

    } catch (error) {
      console.error(`[ImageUploader] 上传失败:`, error);
      return false;
    }
  }

  /**
   * 清除所有参考图
   */
  clear() {
    this.referenceImages.clear();
  }

  /**
   * 获取已加载的参考图数量
   */
  getCount() {
    return this.referenceImages.size;
  }

  /**
   * 获取已加载的角色名列表
   */
  getRoleNames() {
    return Array.from(this.referenceImages.keys());
  }

  delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }
}

// 导出单例
const imageUploader = new ImageUploader();
