/**
 * IndexedDB 存储模块
 * 用于持久化存储参考图（不压缩，保持原图质量）
 * IndexedDB 存储限制远大于 chrome.storage.local（通常几百MB到GB级别）
 */

class IndexedDBStorage {
  constructor() {
    this.dbName = 'JimengReferenceImages';
    this.version = 1;
    this.storeName = 'images';
    this.db = null;
  }

  /**
   * 初始化数据库
   */
  async init() {
    return new Promise((resolve, reject) => {
      const request = indexedDB.open(this.dbName, this.version);

      request.onerror = () => {
        console.error('[IndexedDB] 打开失败:', request.error);
        reject(request.error);
      };

      request.onsuccess = () => {
        this.db = request.result;
        console.log('[IndexedDB] ✅ 数据库已打开');
        resolve();
      };

      request.onupgradeneeded = (event) => {
        const db = event.target.result;

        // 创建对象存储
        if (!db.objectStoreNames.contains(this.storeName)) {
          const objectStore = db.createObjectStore(this.storeName, { keyPath: 'id' });
          objectStore.createIndex('tabName', 'tabName', { unique: false });
          objectStore.createIndex('fileName', 'fileName', { unique: false });
          console.log('[IndexedDB] 对象存储已创建');
        }
      };
    });
  }

  /**
   * 保存图片到指定Tab
   * @param {string} tabName - Tab名称
   * @param {Array<File>} files - 文件列表
   */
  async saveImages(tabName, files) {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readwrite');
    const objectStore = transaction.objectStore(this.storeName);

    const savedFiles = [];

    for (const file of files) {
      const id = `${tabName}_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

      // 读取文件为 ArrayBuffer
      const arrayBuffer = await file.arrayBuffer();

      const imageData = {
        id: id,
        tabName: tabName,
        fileName: file.name,
        fileType: file.type,
        fileSize: file.size,
        data: arrayBuffer,
        timestamp: Date.now()
      };

      try {
        await this.promisifyRequest(objectStore.put(imageData));
        savedFiles.push({
          id: id,
          name: file.name,
          type: file.type,
          size: file.size
        });
        console.log(`[IndexedDB] 保存图片: ${file.name} (${(file.size / 1024).toFixed(1)}KB)`);
      } catch (error) {
        console.error(`[IndexedDB] 保存失败: ${file.name}`, error);
      }
    }

    return savedFiles;
  }

  /**
   * 获取指定Tab的所有图片
   * @param {string} tabName - Tab名称
   * @returns {Array<File>} File对象列表
   */
  async getImages(tabName) {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readonly');
    const objectStore = transaction.objectStore(this.storeName);
    const index = objectStore.index('tabName');

    const request = index.getAll(tabName);

    const results = await this.promisifyRequest(request);

    // 将 ArrayBuffer 转回 File 对象
    const files = results.map(item => {
      const blob = new Blob([item.data], { type: item.fileType });
      return new File([blob], item.fileName, { type: item.fileType });
    });

    console.log(`[IndexedDB] 加载 Tab "${tabName}": ${files.length} 张图片`);
    return files;
  }

  /**
   * 获取所有Tab及其图片信息
   * @returns {Object} { tabName: [{ id, name, type, size }] }
   */
  async getAllTabsInfo() {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readonly');
    const objectStore = transaction.objectStore(this.storeName);

    const request = objectStore.getAll();
    const results = await this.promisifyRequest(request);

    // 按Tab分组
    const tabsInfo = {};
    for (const item of results) {
      if (!tabsInfo[item.tabName]) {
        tabsInfo[item.tabName] = [];
      }
      tabsInfo[item.tabName].push({
        id: item.id,
        name: item.fileName,
        type: item.fileType,
        size: item.fileSize
      });
    }

    return tabsInfo;
  }

  /**
   * 删除指定图片
   * @param {string} imageId - 图片ID
   */
  async deleteImage(imageId) {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readwrite');
    const objectStore = transaction.objectStore(this.storeName);

    await this.promisifyRequest(objectStore.delete(imageId));
    console.log(`[IndexedDB] 删除图片: ${imageId}`);
  }

  /**
   * 删除指定Tab的所有图片
   * @param {string} tabName - Tab名称
   */
  async deleteTab(tabName) {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readwrite');
    const objectStore = transaction.objectStore(this.storeName);
    const index = objectStore.index('tabName');

    const request = index.getAllKeys(tabName);
    const keys = await this.promisifyRequest(request);

    for (const key of keys) {
      await this.promisifyRequest(objectStore.delete(key));
    }

    console.log(`[IndexedDB] 删除 Tab "${tabName}": ${keys.length} 张图片`);
  }

  /**
   * 清空所有数据
   */
  async clearAll() {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readwrite');
    const objectStore = transaction.objectStore(this.storeName);

    await this.promisifyRequest(objectStore.clear());
    console.log('[IndexedDB] 已清空所有数据');
  }

  /**
   * 获取存储使用情况
   * @returns {Object} { totalSize, itemCount }
   */
  async getStorageStats() {
    if (!this.db) await this.init();

    const transaction = this.db.transaction([this.storeName], 'readonly');
    const objectStore = transaction.objectStore(this.storeName);

    const request = objectStore.getAll();
    const results = await this.promisifyRequest(request);

    let totalSize = 0;
    for (const item of results) {
      totalSize += item.fileSize || 0;
    }

    return {
      itemCount: results.length,
      totalSize: totalSize,
      totalSizeMB: (totalSize / (1024 * 1024)).toFixed(2)
    };
  }

  /**
   * 将IndexedDB请求转为Promise
   */
  promisifyRequest(request) {
    return new Promise((resolve, reject) => {
      request.onsuccess = () => resolve(request.result);
      request.onerror = () => reject(request.error);
    });
  }
}

// 导出单例
const indexedDBStorage = new IndexedDBStorage();
