/**
 * 参考图 Tab 管理模块 - FileHandle 版本
 * 使用 FileSystemFileHandle 存储文件引用，保持原图质量
 * 缩略图存储在 chrome.storage 用于UI预览
 */

class ReferenceTabsManager {
  constructor() {
    this.tabs = {}; // { tabName: [{ id, name, type, size, thumbnail, handleId }] }
    this.activeTab = null;
    this.activeTabKey = 'jimeng_active_tab';
    this.thumbnailsKey = 'jimeng_thumbnails'; // 存储缩略图
  }

  /**
   * 初始化 - 从存储加载数据
   */
  async init() {
    try {
      console.log('[ReferenceTabsManager] 开始初始化 (FileHandle 版本)...');

      // 初始化FileHandleStorage
      await fileHandleStorage.init();

      // 获取所有Tab的文件信息
      const tabsInfo = await fileHandleStorage.getAllTabsInfo();

      // 从chrome.storage加载缩略图
      const result = await chrome.storage.local.get([this.activeTabKey, this.thumbnailsKey]);
      const thumbnails = result[this.thumbnailsKey] || {};

      // 重建tabs数据结构
      this.tabs = {};
      for (const [tabName, files] of Object.entries(tabsInfo)) {
        this.tabs[tabName] = files.map(fileInfo => ({
          id: fileInfo.id,
          name: fileInfo.name,
          type: fileInfo.type,
          size: fileInfo.size,
          thumbnail: thumbnails[fileInfo.id] || null,
          handleId: fileInfo.id
        }));

        const totalSize = files.reduce((sum, f) => sum + f.size, 0);
        const sizeMB = (totalSize / (1024 * 1024)).toFixed(2);
        console.log(`[ReferenceTabsManager] Tab "${tabName}": ${files.length} 张图片 (${sizeMB}MB, 原图质量)`);
      }

      // 如果没有Tab，创建默认Tab
      if (Object.keys(this.tabs).length === 0) {
        console.log('[ReferenceTabsManager] 创建默认 Tab');
        this.tabs['默认'] = [];
      }

      // 恢复activeTab
      const savedActiveTab = result[this.activeTabKey];
      if (savedActiveTab && this.tabs[savedActiveTab]) {
        this.activeTab = savedActiveTab;
      } else {
        this.activeTab = Object.keys(this.tabs)[0];
      }

      // 确保activeTab对应的数组存在
      if (!this.tabs[this.activeTab]) {
        console.error('[ReferenceTabsManager] activeTab数组不存在，创建空数组');
        this.tabs[this.activeTab] = [];
      }

      console.log(`[ReferenceTabsManager] ✅ 初始化完成: ${Object.keys(this.tabs).length} 个Tabs`);
      console.log(`[ReferenceTabsManager] 当前Tab: ${this.activeTab}`);

    } catch (error) {
      console.error('[ReferenceTabsManager] ❌ 初始化失败:', error);
      this.tabs = { '默认': [] };
      this.activeTab = '默认';
      throw error;
    }
  }

  /**
   * 添加图片到当前Tab
   * @param {Array<{handle: FileSystemFileHandle, file: File}>} fileHandlePairs - 文件句柄和文件对
   * @returns {number} 添加的数量
   */
  async addImagesToActiveTab(fileHandlePairs) {
    console.log('[ReferenceTabsManager] 开始添加图片...');
    console.log('[ReferenceTabsManager] activeTab:', this.activeTab);
    console.log('[ReferenceTabsManager] 接收到的fileHandlePairs:', fileHandlePairs.length);

    if (!this.activeTab) {
      throw new Error('没有活动的 Tab');
    }

    // 防御性检查
    if (!this.tabs[this.activeTab]) {
      console.error('[ReferenceTabsManager] activeTab数组不存在，创建');
      this.tabs[this.activeTab] = [];
    }

    if (fileHandlePairs.length === 0) {
      throw new Error('没有选择文件');
    }

    try {
      // 1. 保存FileHandle到IndexedDB
      console.log('[ReferenceTabsManager] 保存文件句柄到 IndexedDB...');
      const savedFiles = await fileHandleStorage.saveFileHandles(this.activeTab, fileHandlePairs);

      // 2. 生成缩略图
      console.log('[ReferenceTabsManager] 准备生成缩略图...');

      // 提取File对象
      const files = fileHandlePairs.map(pair => {
        console.log('[ReferenceTabsManager] 提取file:', {
          hasHandle: !!pair.handle,
          hasFile: !!pair.file,
          fileName: pair.file?.name,
          fileType: typeof pair.file,
          isFile: pair.file instanceof File,
          isBlob: pair.file instanceof Blob
        });
        return pair.file;
      });

      console.log('[ReferenceTabsManager] 开始生成缩略图，文件数:', files.length);
      const thumbnailResults = await thumbnailGenerator.generateBatch(files);
      console.log('[ReferenceTabsManager] 缩略图生成完成，结果数:', thumbnailResults.length);

      // 3. 保存缩略图到chrome.storage
      console.log('[ReferenceTabsManager] 保存缩略图到 chrome.storage...');
      const result = await chrome.storage.local.get(this.thumbnailsKey);
      const thumbnails = result[this.thumbnailsKey] || {};

      for (let i = 0; i < savedFiles.length; i++) {
        const savedFile = savedFiles[i];
        const thumbnailResult = thumbnailResults[i];

        if (thumbnailResult) {
          thumbnails[savedFile.id] = thumbnailResult.thumbnail;
        }
      }

      await chrome.storage.local.set({ [this.thumbnailsKey]: thumbnails });

      // 4. 更新内存数据
      for (let i = 0; i < savedFiles.length; i++) {
        const savedFile = savedFiles[i];
        const thumbnailResult = thumbnailResults[i];

        this.tabs[this.activeTab].push({
          id: savedFile.id,
          name: savedFile.name,
          type: savedFile.type,
          size: savedFile.size,
          thumbnail: thumbnailResult ? thumbnailResult.thumbnail : null,
          handleId: savedFile.id
        });
      }

      const totalSize = files.reduce((sum, f) => sum + f.size, 0);
      const sizeMB = (totalSize / (1024 * 1024)).toFixed(2);

      console.log(`[ReferenceTabsManager] ✅ 成功添加 ${savedFiles.length} 张图片 (${sizeMB}MB, 原图质量)`);
      return savedFiles.length;

    } catch (error) {
      console.error('[ReferenceTabsManager] ❌ 添加图片失败:', error);
      console.error('[ReferenceTabsManager] 错误堆栈:', error.stack);
      throw error;
    }
  }

  /**
   * 获取当前Tab的所有原图文件
   * @returns {Promise<Array<File>>} 文件列表
   */
  async getActiveTabFiles() {
    if (!this.activeTab) {
      console.warn('[ReferenceTabsManager] 没有活动的 Tab');
      return [];
    }

    const images = this.tabs[this.activeTab] || [];
    const files = [];
    const failedFiles = [];

    console.log(`[ReferenceTabsManager] 开始读取 ${this.activeTab} 的文件，预期 ${images.length} 个`);

    // 首先处理有 fullData 的图片（外部导入的图片）
    for (const imageInfo of images) {
      if (imageInfo.fullData) {
        try {
          console.log(`[ReferenceTabsManager] 使用 fullData 读取: ${imageInfo.name}`);
          const blob = await this.dataUrlToBlob(imageInfo.fullData);
          const file = new File([blob], `${imageInfo.name}.${imageInfo.type?.split('/')[1] || 'png'}`, {
            type: imageInfo.type || 'image/png'
          });
          files.push(file);
        } catch (error) {
          console.error(`[ReferenceTabsManager] ❌ 读取 fullData 失败: ${imageInfo.name}`, error);
          failedFiles.push(imageInfo.name);
        }
      }
    }

    // 如果已经通过 fullData 获取了图片，直接返回
    if (files.length > 0) {
      console.log(`[ReferenceTabsManager] 从 fullData 获取到 ${files.length} 个文件`);
      return files;
    }

    // 否则从 FileHandleStorage 读取文件句柄
    const handles = await fileHandleStorage.getFileHandles(this.activeTab);

    console.log(`[ReferenceTabsManager] 从 IndexedDB 获取到 ${handles.length} 个句柄`);

    for (const handleData of handles) {
      try {
        if (!handleData.handle) {
          console.warn(`[ReferenceTabsManager] ⚠️ 句柄为空: ${handleData.fileName}`);
          failedFiles.push(handleData.fileName);
          continue;
        }

        // 从handle读取文件（自动验证权限）
        const file = await fileHandleStorage.readFileFromHandle(handleData.handle, handleData);
        if (file) {
          files.push(file);
        } else {
          // 尝试使用备用数据（如果有 fullData）
          const imageInfo = images.find(img => img.handleId === handleData.id);
          if (imageInfo && imageInfo.fullData) {
            console.log(`[ReferenceTabsManager] 使用备用数据: ${handleData.fileName}`);
            const blob = await this.dataUrlToBlob(imageInfo.fullData);
            const backupFile = new File([blob], handleData.fileName, { type: handleData.fileType });
            files.push(backupFile);
          } else {
            console.warn(`[ReferenceTabsManager] ⚠️ 无法读取文件: ${handleData.fileName}`);
            failedFiles.push(handleData.fileName);
          }
        }
      } catch (error) {
        console.error(`[ReferenceTabsManager] ❌ 读取失败: ${handleData.fileName}`, error);
        failedFiles.push(handleData.fileName);
      }
    }

    // 日志输出
    if (failedFiles.length > 0) {
      console.warn(`[ReferenceTabsManager] ⚠️ ${failedFiles.length} 个文件读取失败:`, failedFiles.join(', '));
    }

    console.log(`[ReferenceTabsManager] 获取当前Tab文件: 成功 ${files.length}/${handles.length}`);
    return files;
  }

  /**
   * 预先请求所有文件的权限（在用户手势上下文中调用）
   * @returns {Promise<number>} 成功授权的文件数量
   */
  async requestPermissionsForActiveTab() {
    if (!this.activeTab) {
      return 0;
    }

    const handles = await fileHandleStorage.getFileHandles(this.activeTab);
    let grantedCount = 0;

    console.log(`[ReferenceTabsManager] 开始请求 ${handles.length} 个文件的权限...`);

    for (const handleData of handles) {
      try {
        if (!handleData.handle) continue;

        const permission = await handleData.handle.queryPermission({ mode: 'read' });

        if (permission === 'granted') {
          grantedCount++;
        } else if (permission === 'prompt') {
          // 在用户手势上下文中，这会自动弹出系统权限对话框
          try {
            const newPermission = await handleData.handle.requestPermission({ mode: 'read' });
            if (newPermission === 'granted') {
              grantedCount++;
              console.log(`[ReferenceTabsManager] ✅ 已授权: ${handleData.fileName}`);
            }
          } catch (e) {
            console.warn(`[ReferenceTabsManager] 权限请求失败: ${handleData.fileName}`);
          }
        }
      } catch (error) {
        console.error(`[ReferenceTabsManager] 权限检查失败: ${handleData.fileName}`, error);
      }
    }

    console.log(`[ReferenceTabsManager] 权限请求完成: ${grantedCount}/${handles.length}`);
    return grantedCount;
  }

  /**
   * 将 dataUrl 转换为 Blob
   */
  async dataUrlToBlob(dataUrl) {
    const res = await fetch(dataUrl);
    return res.blob();
  }

  /**
   * 删除指定索引的图片
   * @param {number} index - 索引
   */
  async removeImageFromActiveTab(index) {
    if (!this.activeTab) {
      throw new Error('没有活动的 Tab');
    }

    const images = this.tabs[this.activeTab];
    if (index < 0 || index >= images.length) {
      throw new Error('无效的索引');
    }

    const image = images[index];

    // 从IndexedDB删除句柄
    await fileHandleStorage.deleteHandle(image.handleId);

    // 从chrome.storage删除缩略图
    const result = await chrome.storage.local.get(this.thumbnailsKey);
    const thumbnails = result[this.thumbnailsKey] || {};
    delete thumbnails[image.id];
    await chrome.storage.local.set({ [this.thumbnailsKey]: thumbnails });

    // 从内存删除
    images.splice(index, 1);

    console.log(`[ReferenceTabsManager] 删除图片: ${image.name}`);
  }

  /**
   * 清空当前Tab
   */
  async clearActiveTab() {
    if (!this.activeTab) {
      throw new Error('没有活动的 Tab');
    }

    const count = this.tabs[this.activeTab].length;

    // 从IndexedDB删除
    await fileHandleStorage.deleteTabHandles(this.activeTab);

    // 从chrome.storage删除缩略图
    const result = await chrome.storage.local.get(this.thumbnailsKey);
    const thumbnails = result[this.thumbnailsKey] || {};
    for (const image of this.tabs[this.activeTab]) {
      delete thumbnails[image.id];
    }
    await chrome.storage.local.set({ [this.thumbnailsKey]: thumbnails });

    // 清空内存
    this.tabs[this.activeTab] = [];

    console.log(`[ReferenceTabsManager] 清空 Tab "${this.activeTab}": ${count} 张图片`);
    return count;
  }

  /**
   * 清空所有Tabs
   */
  async clearAllTabs() {
    let totalCount = 0;

    for (const tabName in this.tabs) {
      totalCount += this.tabs[tabName].length;
      await fileHandleStorage.deleteTabHandles(tabName);
      this.tabs[tabName] = [];
    }

    // 清空所有缩略图
    await chrome.storage.local.set({ [this.thumbnailsKey]: {} });

    console.log(`[ReferenceTabsManager] 清空所有 Tabs: ${totalCount} 张图片`);
    return totalCount;
  }

  /**
   * 添加新Tab
   */
  async addTab(tabName) {
    if (!tabName || tabName.trim() === '') {
      throw new Error('Tab 名称不能为空');
    }

    if (this.tabs[tabName]) {
      throw new Error(`Tab "${tabName}" 已存在`);
    }

    this.tabs[tabName] = [];
    this.activeTab = tabName;
    await this.saveActiveTab();
    console.log(`[ReferenceTabsManager] 添加 Tab: ${tabName}`);
  }

  /**
   * 删除Tab
   */
  async deleteTab(tabName) {
    if (!this.tabs[tabName]) {
      throw new Error(`Tab "${tabName}" 不存在`);
    }

    if (Object.keys(this.tabs).length === 1) {
      throw new Error('不能删除最后一个 Tab');
    }

    // 删除该Tab的所有数据
    await fileHandleStorage.deleteTabHandles(tabName);

    // 删除缩略图
    const result = await chrome.storage.local.get(this.thumbnailsKey);
    const thumbnails = result[this.thumbnailsKey] || {};
    for (const image of this.tabs[tabName]) {
      delete thumbnails[image.id];
    }
    await chrome.storage.local.set({ [this.thumbnailsKey]: thumbnails });

    delete this.tabs[tabName];

    // 切换activeTab
    if (this.activeTab === tabName) {
      this.activeTab = Object.keys(this.tabs)[0];
    }

    await this.saveActiveTab();
    console.log(`[ReferenceTabsManager] 删除 Tab: ${tabName}`);
  }

  /**
   * 切换Tab
   */
  async setActiveTab(tabName) {
    if (!this.tabs[tabName]) {
      throw new Error(`Tab "${tabName}" 不存在`);
    }

    this.activeTab = tabName;
    await this.saveActiveTab();
    console.log(`[ReferenceTabsManager] 切换到 Tab: ${tabName}`);
  }

  /**
   * 保存activeTab状态
   */
  async saveActiveTab() {
    await chrome.storage.local.set({ [this.activeTabKey]: this.activeTab });
  }

  // ==================== Getter 方法 ====================

  getActiveTabName() {
    return this.activeTab;
  }

  getActiveTabImages() {
    if (!this.activeTab) return [];
    return this.tabs[this.activeTab] || [];
  }

  getActiveTabImageCount() {
    if (!this.activeTab) return 0;
    return (this.tabs[this.activeTab] || []).length;
  }

  getTabNames() {
    return Object.keys(this.tabs);
  }

  getStats() {
    const stats = {};
    for (const [tabName, images] of Object.entries(this.tabs)) {
      stats[tabName] = {
        count: images.length,
        names: images.map(img => img.name)
      };
    }
    return stats;
  }

  // ==================== 辅助方法 ====================

  /**
   * 诊断并修复
   */
  async diagnoseAndFix() {
    console.log('[ReferenceTabsManager] 🔍 开始诊断...');

    if (!this.activeTab) {
      console.error('[ReferenceTabsManager] ❌ activeTab 为空');
      this.activeTab = '默认';
    }

    if (!this.tabs) {
      console.error('[ReferenceTabsManager] ❌ tabs 对象不存在');
      this.tabs = {};
    }

    if (!this.tabs[this.activeTab]) {
      console.error(`[ReferenceTabsManager] ❌ Tab "${this.activeTab}" 数组不存在`);
      this.tabs[this.activeTab] = [];
    }

    console.log('[ReferenceTabsManager] ✅ 诊断完成');
    console.log('[ReferenceTabsManager] 当前状态:', {
      activeTab: this.activeTab,
      tabs: Object.keys(this.tabs),
      counts: Object.keys(this.tabs).map(t => `${t}: ${this.tabs[t].length}`)
    });
  }

  /**
   * 完全重置
   */
  async reset() {
    console.log('[ReferenceTabsManager] 🔄 开始重置...');

    await fileHandleStorage.clearAll();
    await chrome.storage.local.remove([this.activeTabKey, this.thumbnailsKey]);

    this.tabs = { '默认': [] };
    this.activeTab = '默认';

    console.log('[ReferenceTabsManager] ✅ 重置完成');
  }
}

// 导出单例
const referenceTabsManager = new ReferenceTabsManager();
