/**
 * 角色提取模块
 * 对应 Python 的 extract_roles() 函数
 */

class RoleExtractor {
  /**
   * 从分镜文本中提取角色名称
   * @param {string} text - 分镜文本
   * @returns {Array<string>} 角色列表
   */
  extract(text) {
    const roles = [];

    try {
      // 清理 HTML 标签
      const cleaned = text
        .replace(/<br>/gi, '\n')
        .replace(/<BR>/gi, '\n')
        .replace(/<[^>]+>/g, ''); // 清理所有 HTML 标签

      // 多种正则模式匹配
      const patterns = [
        // 模式1: 到下一个字段为止
        /角色[:：]\s*(.*?)(?=\n\s*(?:机位|构图|姿势|环境|动作|表情|场景|画面|$))/s,
        // 模式2: 到空行为止
        /角色[:：]\s*(.*?)(?=\n\n|\n\[|\Z)/s,
        // 模式3: 仅匹配一行
        /角色[:：]\s*([^\n]+)/
      ];

      let matched = false;

      for (const pattern of patterns) {
        const match = cleaned.match(pattern);

        if (match) {
          const roleText = match[1].trim();
          console.log(`[RoleExtractor] 原始匹配: ${roleText.substring(0, 50)}...`);

          // 检测是否是列表格式 (HTML 格式: * 女儿)
          const listItems = roleText.match(/\*\s*([^\n]+)/g);

          if (listItems) {
            // HTML 列表格式
            for (const item of listItems) {
              const role = item.replace(/\*\s*/, '').trim();
              const cleaned = this.cleanRole(role);
              if (cleaned && !roles.includes(cleaned)) {
                roles.push(cleaned);
              }
            }
          } else {
            // 逗号分隔格式
            const candidates = roleText.split(/[,，、\s/&]+/);
            for (const candidate of candidates) {
              const cleaned = this.cleanRole(candidate);
              if (cleaned && !roles.includes(cleaned)) {
                roles.push(cleaned);
              }
            }
          }

          matched = true;
          break;
        }
      }

      if (matched) {
        console.log(`[RoleExtractor] 提取到角色: ${roles.join(', ')}`);
      } else {
        console.log(`[RoleExtractor] 未找到角色标记`);
      }

    } catch (error) {
      console.error(`[RoleExtractor] 提取失败:`, error);
    }

    return roles;
  }

  /**
   * 清洗角色名称
   * @param {string} role - 原始角色名
   * @returns {string|null} 清洗后的角色名
   */
  cleanRole(role) {
    // 去除首尾空白
    let cleaned = role.trim();

    // 清理 HTML 标签残留
    cleaned = cleaned.replace(/<[^>]+>/g, '');

    // 清理前后缀特殊字符
    cleaned = cleaned.replace(/^[^\w\u4e00-\u9fff]+/, '');
    cleaned = cleaned.replace(/[^\w\u4e00-\u9fff]+$/, '');

    // 过滤无效角色名
    const invalid = ['*', '-', '[主体]', '[环境]', '角色', '空', '无', 'null', 'none'];
    if (invalid.includes(cleaned) || cleaned.length === 0 || /^\d+$/.test(cleaned)) {
      return null;
    }

    return cleaned;
  }
}

// 导出单例
const roleExtractor = new RoleExtractor();
