/**
 * 分镜解析模块
 * 对应 Python 的 parse_storyboards() 和 detect_file_format()
 */

class StoryboardParser {
  /**
   * 智能检测文件格式
   * @param {string} content - 文件内容
   * @returns {'CSV'|'TSV'|'MARKDOWN'|'STORY'} 格式类型
   */
  detectFormat(content) {
    const lines = content.split('\n').filter(line => line.trim());

    if (lines.length === 0) return 'STORY';

    // Markdown 表格特征检测（优先级最高，因为很明确）
    const markdownIndicators = {
      hasPipes: lines[0].includes('|') && lines[0].split('|').length >= 3,
      hasSeparator: lines.length >= 2 && /^\|\s*[:\-\s|]+\s*\|/.test(lines[1]),
      hasHeader: /镜号|分镜|序号|编号/.test(lines[0])
    };

    // CSV 特征检测
    const csvIndicators = {
      hasQuotes: content.match(/"[^"]*"/g)?.length || 0,
      hasCommas: content.split(',').length - 1,
      hasHeader: /分镜数|镜号|分镜|序号|编号/.test(lines[0])
    };

    // TSV 特征检测
    const tsvIndicators = {
      hasTabs: content.split('\t').length - 1,
      hasTabHeader: /镜号|分镜|分镜数|序号/.test(lines[0]) && lines[0].includes('\t')
    };

    // 综合判断（Markdown 优先）
    if (markdownIndicators.hasPipes && markdownIndicators.hasSeparator) {
      return 'MARKDOWN';
    } else if (csvIndicators.hasQuotes >= 2 && csvIndicators.hasCommas >= 5) {
      return 'CSV';
    } else if (tsvIndicators.hasTabs >= 3 || tsvIndicators.hasTabHeader) {
      return 'TSV';
    } else {
      return 'STORY';
    }
  }

  /**
   * 解析 CSV 格式
   * @param {string} content - CSV 内容
   * @returns {Array<string>} 分镜列表
   */
  parseCSV(content) {
    // 使用 PapaParse 库
    const result = Papa.parse(content, {
      header: true,
      skipEmptyLines: 'greedy', // 只跳过完全空白的行，保留内容中的空行
      dynamicTyping: false,
      newline: '', // 自动检测换行符
      quoteChar: '"',
      escapeChar: '"'
    });

    console.log(`[StoryboardParser] CSV解析结果: ${result.data.length} 行数据`);
    console.log(`[StoryboardParser] CSV列名:`, result.meta.fields);

    const storyboards = [];
    const columnNames = ['分镜提示词', '分镜', '提示词', 'content', '内容', 'prompt', 'text'];

    // 遍历每一行数据
    for (let i = 0; i < result.data.length; i++) {
      const row = result.data[i];

      // 跳过完全空的行
      if (!row || Object.keys(row).length === 0) {
        console.log(`[StoryboardParser] 跳过空行: 第${i+1}行`);
        continue;
      }

      // 查找包含分镜内容的列
      let content = null;

      // 方式1: 尝试预定义的列名
      for (const colName of columnNames) {
        if (row[colName] && typeof row[colName] === 'string') {
          const trimmed = row[colName].trim();
          if (trimmed.length > 5) {
            content = trimmed;
            console.log(`[StoryboardParser] 第${i+1}行: 使用列"${colName}", 长度${trimmed.length}`);
            break;
          }
        }
      }

      // 方式2: 如果没找到，尝试所有列（选择最长的内容）
      if (!content) {
        let maxContent = '';
        for (const [key, value] of Object.entries(row)) {
          if (value && typeof value === 'string') {
            const trimmed = value.trim();
            if (trimmed.length > maxContent.length && trimmed.length > 5) {
              maxContent = trimmed;
              console.log(`[StoryboardParser] 第${i+1}行: 备用列"${key}", 长度${trimmed.length}`);
            }
          }
        }
        if (maxContent) {
          content = maxContent;
        }
      }

      if (content) {
        storyboards.push(content);
      } else {
        console.warn(`[StoryboardParser] ⚠️ 第${i+1}行: 未找到有效内容`, row);
      }
    }

    console.log(`[StoryboardParser] CSV解析完成: 找到 ${storyboards.length} 个分镜`);

    // 显示前3个分镜的预览（用于调试）
    if (storyboards.length > 0) {
      console.log(`[StoryboardParser] 分镜预览 (前3个):`);
      storyboards.slice(0, 3).forEach((sb, i) => {
        console.log(`  #${i+1}: ${sb.substring(0, 50)}...`);
      });
    }

    return storyboards;
  }

  /**
   * 解析 TSV 格式
   * @param {string} content - TSV 内容
   * @returns {Array<string>} 分镜列表
   */
  parseTSV(content) {
    const lines = content.split('\n');
    const storyboards = [];

    for (let i = 0; i < lines.length; i++) {
      const line = lines[i].trim();

      // 跳过空行和表头
      if (!line || (i === 0 && /镜号|分镜|序号/.test(line))) {
        continue;
      }

      const parts = line.split('\t');
      if (parts.length >= 2) {
        const content = parts[1].trim();
        if (content.length > 5) {
          storyboards.push(content);
        }
      }
    }

    console.log(`[StoryboardParser] TSV解析: 找到 ${storyboards.length} 个分镜`);
    return storyboards;
  }

  /**
   * 解析 Markdown 表格格式
   * @param {string} content - Markdown 表格内容
   * @returns {Array<string>} 分镜列表
   */
  parseMarkdownTable(content) {
    const lines = content.split('\n');
    const storyboards = [];

    for (let i = 0; i < lines.length; i++) {
      const line = lines[i].trim();

      // 跳过空行、表头和分隔线
      if (!line ||
          line.includes('镜号') ||
          line.includes('分镜提示词') ||
          /^\|\s*[:\-\s|]+\s*\|/.test(line)) {
        continue;
      }

      // 解析表格行
      if (line.startsWith('|') && line.endsWith('|')) {
        // 分割管道符，去除首尾空元素
        const parts = line.split('|').map(p => p.trim()).filter(p => p);

        // 至少有两列（镜号 + 内容）
        if (parts.length >= 2) {
          // 第二列是分镜内容
          let content = parts[1];

          // 清理 HTML 标签和 Markdown 格式
          content = content
            .replace(/<br\s*\/?>/gi, '\n')           // <br> 转换为换行
            .replace(/\\\*/g, '*')                    // \* 转换为 *
            .replace(/\*\*([^*]+)\*\*/g, '$1')       // **文字** 转为 文字
            .replace(/\* /g, '')                      // 移除列表符号
            .replace(/<[^>]+>/g, '')                  // 移除其他 HTML 标签
            .trim();

          // 至少 10 个字符才算有效
          if (content.length > 10) {
            storyboards.push(content);
          }
        }
      }
    }

    console.log(`[StoryboardParser] Markdown表格解析: 找到 ${storyboards.length} 个分镜`);
    return storyboards;
  }

  /**
   * 解析故事格式
   * @param {string} content - 故事格式内容
   * @returns {Array<string>} 分镜列表
   */
  parseStory(content) {
    const storyboards = [];

    // 正则匹配故事分隔符
    const separatorPattern = /^(故事组?[\d一二三四五六七八九十百千万]+.*|故事\s*[\d一二三四五六七八九十百千万]+\s*[：:].*)$/gm;

    const parts = content.split(separatorPattern);

    for (let i = 0; i < parts.length; i++) {
      const part = parts[i]?.trim();

      // 跳过分隔符本身和空内容
      if (!part || separatorPattern.test(part)) {
        continue;
      }

      // 至少 20 个字符才算有效
      if (part.length > 20) {
        storyboards.push(part);
      }
    }

    console.log(`[StoryboardParser] 故事格式解析: 找到 ${storyboards.length} 个分镜`);
    return storyboards;
  }

  /**
   * 统一解析入口
   * @param {string} content - 文件内容
   * @returns {Array<string>} 分镜列表
   */
  parse(content) {
    const format = this.detectFormat(content);
    console.log(`[StoryboardParser] 检测到格式: ${format}`);

    let storyboards = [];

    switch (format) {
      case 'CSV':
        storyboards = this.parseCSV(content);
        break;
      case 'TSV':
        storyboards = this.parseTSV(content);
        break;
      case 'MARKDOWN':
        storyboards = this.parseMarkdownTable(content);
        break;
      case 'STORY':
        storyboards = this.parseStory(content);
        break;
      default:
        storyboards = [];
    }

    return this.addSequenceNumbers(storyboards);
  }

  /**
   * 为每个分镜内容添加序号前缀 (例如: 1.XXXX)
   * 若内容已带数字序号则保持不变，避免重复前缀
   * @param {Array<string>} storyboards - 原始分镜列表
   * @returns {Array<string>} 带序号的分镜列表
   */
  addSequenceNumbers(storyboards) {
    const numberPrefixPattern = /^\s*\d+[\.\uff0e、:：-]/;

    return storyboards.map((content, idx) => {
      const trimmed = (content || '').trim();
      if (!trimmed) return trimmed;

      if (numberPrefixPattern.test(trimmed)) {
        return trimmed;
      }

      return `${idx + 1}.${trimmed}`;
    });
  }
}

// 导出单例
const storyboardParser = new StoryboardParser();
