/**
 * 提交处理模块
 * 对应 Python 的 process_one_storyboard()、input_content()、submit() 等
 */

class SubmitHandler {
  constructor() {
    this.isRunning = false;
    this.currentIndex = 0;
    this.maxRetries = 2;
    this.settings = {
      inputDelay: 2000,
      submitDelay: 3000,
      statusCheckTimeout: 15000  // 增加到15秒,给即梦更多响应时间
    };
  }

  /**
   * 处理单个分镜 (完整流程)
   * @param {string} storyboard - 分镜内容
   * @param {number} index - 当前索引 (1-based)
   * @param {number} total - 总数
   * @returns {Promise<boolean>} 是否成功
   */
  async processOne(storyboard, index, total) {
    console.log(`[${index}/${total}] 开始处理分镜`);

    // 重试循环
    for (let attempt = 0; attempt <= this.maxRetries; attempt++) {
      try {
        if (attempt > 0) {
          console.log(`🔄 [${index}/${total}] 第 ${attempt} 次重试`);
          await this.delay(3000);
        }

        // 1. 清除旧参考图
        await this.clearReferenceImages();

        // 2. 提取角色
        const roles = roleExtractor.extract(storyboard);

        // 3. 上传参考图
        if (roles.length > 0) {
          const uploaded = await imageUploader.uploadForRoles(roles);
          if (!uploaded) {
            console.warn(`⚠️ [${index}/${total}] 参考图上传失败,继续处理`);
          }
        }

        // 4. 输入分镜内容
        const inputSuccess = await this.inputContent(storyboard);
        if (!inputSuccess) {
          console.error(`❌ [${index}/${total}] 输入失败`);
          if (attempt < this.maxRetries) continue;
          return false;
        }

        // 5. 点击提交按钮
        const clickSuccess = await this.clickSubmit();
        if (!clickSuccess) {
          console.error(`❌ [${index}/${total}] 点击提交失败`);
          if (attempt < this.maxRetries) continue;
          return false;
        }

        // 6. 检查提交状态
        const statusOk = await this.checkSubmitStatus();
        if (statusOk) {
          console.log(`✅ [${index}/${total}] 分镜处理成功`);

          // 提交后等待
          await this.delay(this.settings.submitDelay);

          return true;
        } else {
          console.error(`❌ [${index}/${total}] 提交状态检测失败`);
          if (attempt < this.maxRetries) continue;
          return false;
        }

      } catch (error) {
        console.error(`❌ [${index}/${total}] 异常:`, error);
        if (attempt < this.maxRetries) continue;
        return false;
      }
    }

    return false;
  }

  /**
   * 清除旧参考图 (DOM 操作)
   */
  async clearReferenceImages() {
    const selectors = [
      'div.reference-item-OOc16S',
      'div[class*="reference-item"]',
      'div[class*="image-item"]'
    ];

    let deletedCount = 0;

    for (const selector of selectors) {
      const elements = document.querySelectorAll(selector);

      for (const el of elements) {
        // 检查是否包含文件上传控件 (不删除上传按钮)
        const hasFileInput = el.querySelector('input[type="file"]');
        if (hasFileInput) continue;

        // 尝试点击删除按钮
        const deleteBtn = el.querySelector('[class*="delete"], [class*="remove"], [class*="close"]');
        if (deleteBtn) {
          deleteBtn.click();
          deletedCount++;
        } else {
          // 直接删除 DOM
          el.remove();
          deletedCount++;
        }
      }

      if (deletedCount > 0) break;
    }

    if (deletedCount > 0) {
      console.log(`[SubmitHandler] 清除了 ${deletedCount} 张旧参考图`);
      await this.delay(500);
    }
  }

  /**
   * 输入分镜内容
   * @param {string} text - 分镜文本
   * @returns {Promise<boolean>} 是否成功
   */
  async inputContent(text) {
    const selectors = [
      "textarea[placeholder*='图片']",
      "textarea[placeholder*='提示']",
      "textarea.prompt-textarea-XfqAoB",
      "textarea"
    ];

    for (const selector of selectors) {
      const textareas = document.querySelectorAll(selector);

      for (const textarea of textareas) {
        // 检查是否可见
        if (textarea.offsetParent === null) continue;

        try {
          // 聚焦
          textarea.focus();
          await this.delay(100);

          // 清空 (全选+删除)
          textarea.select();
          document.execCommand('delete');

          textarea.value = '';
          textarea.dispatchEvent(new Event('input', { bubbles: true }));

          await this.delay(100);

          // 填充内容
          textarea.value = text;
          textarea.dispatchEvent(new Event('input', { bubbles: true }));
          textarea.dispatchEvent(new Event('change', { bubbles: true }));

          // 触发焦点事件
          textarea.blur();
          textarea.focus();

          console.log(`[SubmitHandler] 输入成功 (${text.length} 字符)`);

          await this.delay(this.settings.inputDelay);

          return true;

        } catch (error) {
          console.warn(`[SubmitHandler] 输入失败:`, error);
          continue;
        }
      }
    }

    console.error(`[SubmitHandler] 未找到可用的输入框`);
    return false;
  }

  /**
   * 点击提交按钮
   * @returns {Promise<boolean>} 是否成功
   */
  async clickSubmit() {
    const selectors = [
      "button.submit-button-VW0U_J",
      "button.lv-btn-primary",
      ".lv-btn.lv-btn-primary.submit-button-VW0U_J",
      "button[type='submit']"
    ];

    for (const selector of selectors) {
      const buttons = document.querySelectorAll(selector);

      for (const button of buttons) {
        try {
          // 检查是否禁用
          const className = button.className || '';
          if (className.includes('disabled') || button.disabled) {
            console.log(`[SubmitHandler] 按钮禁用,跳过`);
            continue;
          }

          // 检查是否可见
          if (button.offsetParent === null) continue;

          // 滚动到视图
          button.scrollIntoView({ block: 'center', behavior: 'smooth' });
          await this.delay(300);

          // 尝试多种点击方式
          let clicked = false;

          // 方式1: 正常点击
          try {
            button.click();
            clicked = true;
          } catch (e) {
            console.warn(`[SubmitHandler] 正常点击失败`, e);
          }

          // 方式2: 事件模拟
          if (!clicked) {
            try {
              const event = new MouseEvent('click', {
                bubbles: true,
                cancelable: true,
                view: window
              });
              button.dispatchEvent(event);
              clicked = true;
            } catch (e) {
              console.warn(`[SubmitHandler] 事件点击失败`, e);
            }
          }

          if (clicked) {
            console.log(`[SubmitHandler] 点击提交按钮成功`);
            return true;
          }

        } catch (error) {
          console.warn(`[SubmitHandler] 点击失败:`, error);
          continue;
        }
      }
    }

    console.error(`[SubmitHandler] 未找到可点击的提交按钮`);
    return false;
  }

  /**
   * 检查提交状态 (监听 Toast 提示)
   * @returns {Promise<boolean>} 提交是否成功
   */
  async checkSubmitStatus() {
    return new Promise((resolve) => {
      let resolved = false;
      let lastToastText = '';

      const timeout = setTimeout(() => {
        if (!resolved) {
          resolved = true;
          observer.disconnect();

          // 超时时检查按钮状态作为后备方案
          const submitBtn = this.findSubmitButton();
          if (submitBtn && !submitBtn.disabled && !submitBtn.className.includes('disabled')) {
            // 按钮可用,说明可能已经提交成功了
            console.log(`[SubmitHandler] 状态检查超时,但提交按钮可用,判定为成功`);
            resolve(true);
          } else {
            console.warn(`[SubmitHandler] 状态检查超时,最后提示: ${lastToastText || '无'}`);
            // 如果没有检测到明确的失败提示,默认视为成功
            resolve(true);
          }
        }
      }, this.settings.statusCheckTimeout);

      // 监听 DOM 变化 (Toast 提示)
      const observer = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
          if (mutation.type === 'childList') {
            mutation.addedNodes.forEach(node => {
              if (node.nodeType !== 1) return;

              const text = node.textContent || '';

              // 记录最后的 Toast 文本
              if (text.length > 2 && text.length < 100) {
                lastToastText = text;
              }

              // 失败关键词 - 只在明确失败时才判定失败
              const failKeywords = ['提交失败', '生成失败', '错误', '违规', '操作过于频繁', '稍后再试', '系统异常', '配额不足'];
              if (failKeywords.some(kw => text.includes(kw))) {
                if (!resolved) {
                  resolved = true;
                  clearTimeout(timeout);
                  observer.disconnect();
                  console.error(`[SubmitHandler] 检测到失败提示: ${text}`);
                  resolve(false);
                }
              }

              // 成功关键词
              const successKeywords = ['生成中', '排队中', '已提交', '提交成功', '已接收', '开始生成', '任务已创建'];
              if (successKeywords.some(kw => text.includes(kw))) {
                if (!resolved) {
                  resolved = true;
                  clearTimeout(timeout);
                  observer.disconnect();
                  console.log(`[SubmitHandler] 检测到成功提示: ${text}`);
                  resolve(true);
                }
              }
            });
          }
        }
      });

      // 开始监听
      observer.observe(document.body, {
        childList: true,
        subtree: true
      });
    });
  }

  /**
   * 查找提交按钮 (辅助方法)
   */
  findSubmitButton() {
    const selectors = [
      "button.submit-button-VW0U_J",
      "button.lv-btn-primary",
      ".lv-btn.lv-btn-primary.submit-button-VW0U_J",
      "button[type='submit']"
    ];

    for (const selector of selectors) {
      const buttons = document.querySelectorAll(selector);
      for (const button of buttons) {
        if (button.offsetParent !== null) {
          return button;
        }
      }
    }
    return null;
  }

  delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }
}

// 导出单例
const submitHandler = new SubmitHandler();
