/**
 * 缩略图生成器
 * 用于生成小尺寸预览图，存储到chrome.storage
 */

class ThumbnailGenerator {
  constructor() {
    this.thumbnailSize = 100; // 缩略图大小 100x100
    this.quality = 0.8; // 缩略图质量
  }

  /**
   * 从File对象生成缩略图
   * @param {File} file - 图片文件
   * @returns {Promise<string>} Base64格式的缩略图
   */
  async generateFromFile(file) {
    return new Promise((resolve, reject) => {
      // 验证输入
      if (!file || !(file instanceof File || file instanceof Blob)) {
        console.error('[ThumbnailGenerator] 无效的文件对象:', file);
        reject(new Error('无效的文件对象'));
        return;
      }

      const reader = new FileReader();

      reader.onload = (e) => {
        const img = new Image();

        img.onload = () => {
          try {
            const thumbnail = this.createThumbnail(img);
            resolve(thumbnail);
          } catch (error) {
            console.error('[ThumbnailGenerator] 创建缩略图失败:', error);
            reject(error);
          }
        };

        img.onerror = () => {
          reject(new Error('图片加载失败'));
        };

        img.src = e.target.result;
      };

      reader.onerror = () => {
        reject(new Error('文件读取失败'));
      };

      try {
        reader.readAsDataURL(file);
      } catch (error) {
        console.error('[ThumbnailGenerator] FileReader错误:', error);
        reject(error);
      }
    });
  }

  /**
   * 创建缩略图
   * @param {HTMLImageElement} img - 图片元素
   * @returns {string} Base64格式的缩略图
   */
  createThumbnail(img) {
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');

    // 计算缩放后的尺寸（保持宽高比）
    let width = img.width;
    let height = img.height;
    const maxSize = this.thumbnailSize;

    if (width > height) {
      if (width > maxSize) {
        height = (height * maxSize) / width;
        width = maxSize;
      }
    } else {
      if (height > maxSize) {
        width = (width * maxSize) / height;
        height = maxSize;
      }
    }

    canvas.width = width;
    canvas.height = height;

    // 绘制缩略图
    ctx.drawImage(img, 0, 0, width, height);

    // 转换为Base64
    return canvas.toDataURL('image/jpeg', this.quality);
  }

  /**
   * 批量生成缩略图
   * @param {Array<File>} files - 文件列表
   * @returns {Promise<Array<{file: File, thumbnail: string}>>}
   */
  async generateBatch(files) {
    console.log(`[ThumbnailGenerator] 开始批量生成 ${files.length} 个缩略图`);

    const results = [];

    for (let i = 0; i < files.length; i++) {
      const file = files[i];

      try {
        // 验证文件对象
        if (!file) {
          console.error(`[ThumbnailGenerator] 第${i+1}个文件为null`);
          continue;
        }

        console.log(`[ThumbnailGenerator] 处理文件 ${i+1}/${files.length}: ${file.name}, 类型: ${file.type}, 大小: ${file.size}`);

        const thumbnail = await this.generateFromFile(file);

        results.push({
          file: file,
          thumbnail: thumbnail,
          name: file.name,
          type: file.type,
          size: file.size
        });

        console.log(`[ThumbnailGenerator] ✅ 生成缩略图: ${file.name}`);
      } catch (error) {
        console.error(`[ThumbnailGenerator] ❌ 生成失败: ${file ? file.name : 'unknown'}`, error);
        console.error(`[ThumbnailGenerator] 错误详情:`, {
          file: file,
          fileType: typeof file,
          isFile: file instanceof File,
          isBlob: file instanceof Blob,
          error: error.message
        });
      }
    }

    console.log(`[ThumbnailGenerator] 批量生成完成: ${results.length}/${files.length}`);
    return results;
  }
}

// 导出单例
const thumbnailGenerator = new ThumbnailGenerator();
