/**
 * Popup Script - 弹窗界面逻辑
 */

console.log('[Popup] 初始化');

// 全局变量
let fileContent = null;
let storyboardCount = 0;
let referenceFiles = [];
let logs = [];

// DOM 元素
const fileUploadArea = document.getElementById('fileUploadArea');
const fileInput = document.getElementById('fileInput');
const uploadPlaceholder = document.getElementById('uploadPlaceholder');
const fileInfo = document.getElementById('fileInfo');
const fileName = document.getElementById('fileName');
const fileCount = document.getElementById('fileCount');
const clearFileBtn = document.getElementById('clearFileBtn');

const refInput = document.getElementById('refInput');
const selectRefBtn = document.getElementById('selectRefBtn');
const refCount = document.getElementById('refCount');
const refList = document.getElementById('refList');

const startIndexInput = document.getElementById('startIndex');
const endIndexInput = document.getElementById('endIndex');
const delayInput = document.getElementById('delay');
const retryTimesInput = document.getElementById('retryTimes');
const autoDownloadCheckbox = document.getElementById('autoDownload');

const startBtn = document.getElementById('startBtn');
const pauseBtn = document.getElementById('pauseBtn');

const progressSection = document.getElementById('progressSection');
const progressBar = document.getElementById('progressBar');
const progressText = document.getElementById('progressText');
const progressDetail = document.getElementById('progressDetail');
const progressEta = document.getElementById('progressEta');

const logContainer = document.getElementById('logContainer');
const clearLogBtn = document.getElementById('clearLogBtn');
const exportLogBtn = document.getElementById('exportLogBtn');

// 文件上传区域点击事件
fileUploadArea.addEventListener('click', () => {
  fileInput.click();
});

// 文件选择事件
fileInput.addEventListener('change', async (e) => {
  const file = e.target.files[0];
  if (!file) return;

  try {
    fileContent = await file.text();

    // 简单计算分镜数量
    storyboardCount = estimateStoryboardCount(fileContent);

    // 更新UI
    fileName.textContent = file.name;
    fileCount.textContent = `(${storyboardCount} 个分镜)`;
    uploadPlaceholder.style.display = 'none';
    fileInfo.style.display = 'flex';

    // 更新结束索引
    endIndexInput.value = storyboardCount;

    appendLog(`已加载文件: ${file.name}, 检测到 ${storyboardCount} 个分镜`);
  } catch (error) {
    appendLog(`❌ 文件读取失败: ${error.message}`);
  }
});

// 清除文件按钮
clearFileBtn.addEventListener('click', (e) => {
  e.stopPropagation();
  fileContent = null;
  storyboardCount = 0;
  fileInput.value = '';
  uploadPlaceholder.style.display = 'block';
  fileInfo.style.display = 'none';
  appendLog('已清除文件');
});

// 参考图选择按钮
selectRefBtn.addEventListener('click', () => {
  refInput.click();
});

// 参考图选择事件
refInput.addEventListener('change', (e) => {
  referenceFiles = Array.from(e.target.files);
  refCount.textContent = `已选: ${referenceFiles.length} 张`;
  updateRefList();
  appendLog(`已加载 ${referenceFiles.length} 张参考图`);
});

// 更新参考图列表
function updateRefList() {
  refList.innerHTML = '';
  referenceFiles.forEach((file, index) => {
    const item = document.createElement('div');
    item.className = 'ref-item';
    item.innerHTML = `
      <span>✅ ${file.name}</span>
      <span class="file-size">${formatFileSize(file.size)}</span>
      <button class="btn-small" onclick="removeRef(${index})">删除</button>
    `;
    refList.appendChild(item);
  });
}

// 删除参考图
window.removeRef = (index) => {
  referenceFiles.splice(index, 1);
  refCount.textContent = `已选: ${referenceFiles.length} 张`;
  updateRefList();
};

// 开始批量按钮
startBtn.addEventListener('click', async () => {
  if (!fileContent) {
    alert('请先选择分镜文件');
    return;
  }

  // 转换参考图为 Base64
  const refImagesBase64 = {};
  for (const file of referenceFiles) {
    const base64 = await fileToBase64(file);
    const name = file.name.replace(/\.\w+$/, '');
    refImagesBase64[name] = { data: base64, name: file.name };
  }

  // 获取设置
  const settings = {
    startIndex: parseInt(startIndexInput.value),
    endIndex: parseInt(endIndexInput.value),
    retryTimes: parseInt(retryTimesInput.value),
    delay: parseInt(delayInput.value) * 1000,
    autoDownload: autoDownloadCheckbox.checked
  };

  // 发送消息到 content.js
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (!tabs[0]) {
      alert('请先打开即梦网页');
      return;
    }

    if (!tabs[0].url.includes('jimeng.jianying.com')) {
      alert('请在即梦网页上使用此插件');
      return;
    }

    chrome.tabs.sendMessage(tabs[0].id, {
      action: 'startBatch',
      data: {
        content: fileContent,
        referenceImages: refImagesBase64,
        settings: settings
      }
    }, (response) => {
      if (chrome.runtime.lastError) {
        console.error('发送失败:', chrome.runtime.lastError);
        alert('请确保已打开即梦网页并刷新页面');
      } else {
        appendLog('✅ 任务已发送,开始执行...');
        progressSection.style.display = 'block';
        startBtn.disabled = true;
        pauseBtn.disabled = false;
      }
    });
  });
});

// 暂停按钮
pauseBtn.addEventListener('click', () => {
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    chrome.tabs.sendMessage(tabs[0].id, { action: 'pauseBatch' });
    appendLog('⏸️ 任务已暂停');
  });
});

// 清空日志
clearLogBtn.addEventListener('click', () => {
  logs = [];
  logContainer.innerHTML = '';
  appendLog('日志已清空');
});

// 导出日志
exportLogBtn.addEventListener('click', () => {
  const logText = logs.map(log => `${log.time} ${log.message}`).join('\n');
  const blob = new Blob([logText], { type: 'text/plain' });
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = `即梦批量助手_日志_${new Date().toISOString().slice(0, 10)}.txt`;
  a.click();
  URL.revokeObjectURL(url);
});

// 监听来自 content.js 的消息
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'updateProgress') {
    updateProgress(message.current, message.total);
    appendLog(message.log);
  }

  if (message.action === 'batchComplete') {
    const result = message.result;
    appendLog(`✅ 批量任务完成! 成功: ${result.success}, 失败: ${result.failed}`);
    if (result.failedIndices.length > 0) {
      appendLog(`失败的分镜索引: ${result.failedIndices.join(', ')}`);
    }
    startBtn.disabled = false;
    pauseBtn.disabled = true;
  }

  if (message.action === 'updateImageCount') {
    appendLog(`📷 检测到 ${message.count} 张生成的图片`);
  }
});

// 更新进度条
function updateProgress(current, total) {
  if (total === 0) return;

  const percent = Math.round((current / total) * 100);
  progressBar.style.width = percent + '%';
  progressText.textContent = percent + '%';
  progressDetail.textContent = `${current}/${total}`;
}

// 添加日志
function appendLog(message) {
  const time = new Date().toLocaleTimeString();
  logs.push({ time, message });

  // 保持最多20条
  if (logs.length > 20) {
    logs.shift();
  }

  const entry = document.createElement('div');
  entry.className = 'log-entry';

  // 根据消息类型设置样式
  let className = '';
  if (message.includes('✅') || message.includes('成功')) {
    className = 'log-success';
  } else if (message.includes('❌') || message.includes('失败')) {
    className = 'log-error';
  } else if (message.includes('⚠️') || message.includes('警告')) {
    className = 'log-warning';
  }

  entry.innerHTML = `
    <span class="log-time">[${time}]</span>
    <span class="log-message ${className}">${message}</span>
  `;

  logContainer.appendChild(entry);
  logContainer.scrollTop = logContainer.scrollHeight;
}

// 估算分镜数量
function estimateStoryboardCount(content) {
  // 简单估算,根据格式判断
  if (content.includes(',') && content.includes('"')) {
    // CSV 格式
    const lines = content.split('\n').filter(line => line.trim());
    return Math.max(0, lines.length - 1); // 减去表头
  } else if (content.includes('\t')) {
    // TSV 格式
    const lines = content.split('\n').filter(line => line.trim());
    return Math.max(0, lines.length - 1);
  } else {
    // 故事格式
    const matches = content.match(/故事组?[\d一二三四五六七八九十百千万]+/g);
    return matches ? matches.length : 1;
  }
}

// File 转 Base64
function fileToBase64(file) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onload = () => resolve(reader.result);
    reader.onerror = reject;
    reader.readAsDataURL(file);
  });
}

// 格式化文件大小
function formatFileSize(bytes) {
  if (bytes < 1024) return bytes + ' B';
  if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
  return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
}

// 页面加载时恢复设置
chrome.storage.local.get(['settings', 'logs'], (result) => {
  if (result.settings) {
    const settings = result.settings;
    if (settings.startIndex) startIndexInput.value = settings.startIndex;
    if (settings.endIndex) endIndexInput.value = settings.endIndex;
    if (settings.delay) delayInput.value = settings.delay;
    if (settings.retryTimes) retryTimesInput.value = settings.retryTimes;
    if (settings.autoDownload !== undefined) autoDownloadCheckbox.checked = settings.autoDownload;
  }

  if (result.logs && result.logs.length > 0) {
    logs = result.logs;
    logs.forEach(log => {
      const entry = document.createElement('div');
      entry.className = 'log-entry';
      entry.innerHTML = `
        <span class="log-time">[${log.time}]</span>
        <span class="log-message">${log.message}</span>
      `;
      logContainer.appendChild(entry);
    });
  }
});

// 页面关闭时保存设置
window.addEventListener('beforeunload', () => {
  const settings = {
    startIndex: startIndexInput.value,
    endIndex: endIndexInput.value,
    delay: delayInput.value,
    retryTimes: retryTimesInput.value,
    autoDownload: autoDownloadCheckbox.checked
  };

  chrome.storage.local.set({ settings, logs });
});
