// ==================== 扩展弹窗脚本 ====================

document.addEventListener('DOMContentLoaded', async () => {
    const scrapeBtn = document.getElementById('scrapeBtn');
    const totalCountEl = document.getElementById('totalCount');
    const lastUpdateEl = document.getElementById('lastUpdate');
    const sourcesListEl = document.getElementById('sourcesList');

    // 加载上次抓取的数据
    const stored = await chrome.storage.local.get('lastScrape');
    if (stored.lastScrape) {
        updateUI(stored.lastScrape);
    } else {
        // 显示默认的源列表
        const sources = await chrome.runtime.sendMessage({ action: 'getSources' });
        if (sources) {
            renderSourcesList(sources.map(s => ({
                id: s.id,
                name: s.name,
                count: '-',
                needsLogin: s.needsLogin
            })));
        }
    }

    // 点击抓取按钮
    scrapeBtn.addEventListener('click', async () => {
        scrapeBtn.disabled = true;
        scrapeBtn.textContent = '⏳ 抓取中...';

        try {
            const response = await chrome.runtime.sendMessage({ action: 'scrapeAll' });
            updateUI(response);
            scrapeBtn.textContent = '✅ 抓取完成！';

            setTimeout(() => {
                scrapeBtn.disabled = false;
                scrapeBtn.textContent = '🚀 开始抓取（5个网站）';
            }, 2000);
        } catch (error) {
            scrapeBtn.textContent = '❌ 抓取失败';
            console.error(error);

            setTimeout(() => {
                scrapeBtn.disabled = false;
                scrapeBtn.textContent = '🚀 重新抓取';
            }, 2000);
        }
    });

    function updateUI(data) {
        totalCountEl.textContent = data.totalCount || data.videos?.length || 0;
        if (data.timestamp) {
            lastUpdateEl.textContent = new Date(data.timestamp).toLocaleTimeString('zh-CN');
        }

        // 渲染各网站状态
        if (data.sources) {
            const sourcesList = Object.entries(data.sources).map(([id, info]) => ({
                id,
                name: info.name || id,
                count: info.success ? info.count : 'error',
                error: info.error
            }));
            renderSourcesList(sourcesList);
        }
    }

    function renderSourcesList(sources) {
        sourcesListEl.innerHTML = sources.map(s => `
            <div class="source-item">
                <span class="source-name">${s.name}</span>
                ${s.count === 'error' || s.count === '-'
                ? `<span class="source-error">${s.error || (s.needsLogin ? '需登录' : '待抓取')}</span>`
                : `<span class="source-count">${s.count} 个</span>`
            }
            </div>
        `).join('');
    }
});
